#pragma once

#include <algorithm>
#include <array>
#include <optional>
#include <variant>
#include <iterator>
#include <map>

#include <boost/algorithm/searching/boyer_moore.hpp>
#include <boost/range/adaptor/adjacent_filtered.hpp>
#include <boost/range/adaptor/replaced_if.hpp>
#include <boost/range/adaptor/replaced.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/case_conv.hpp>

namespace unistat {

template<typename T = double>
using NamedValue = std::tuple<std::string, T>;

template<typename BucketBound = double>
using NamedHist = std::tuple<std::string, std::vector<std::tuple<BucketBound, std::size_t>>>;


inline std::string withSigoptSuffix(std::string_view value, std::string_view suffix) {
    if (!value.empty() && value.back() == '_' || !suffix.empty() && suffix.front() == '_') {
        return std::string(value) + std::string(suffix);
    }
    return std::string(value) + "_" + std::string(suffix);
}

inline std::string buildSignalName(std::string_view name, std::string_view endpoint) {
    if (name.empty() && endpoint.empty()) {
        return "";
    }
    if (endpoint.empty()) {
        return std::string(name);
    }
    return std::string(name) + "_" + std::string(endpoint);
}

inline std::string normalizeName(std::string name) {
    using namespace boost::adaptors;
    using namespace boost::algorithm;

    constexpr unsigned int maxSignalNameLength = 128;

    to_lower(name);
    const auto from = name | replaced_if(is_any_of("= ')(?/"), '_')
            | replaced(':', '_')
            | adjacent_filtered([] (auto lhs, auto rhs) {
        return '_' != lhs || '_' != rhs;
    });

    name.assign(from.begin(), from.end());
    if (maxSignalNameLength < name.size()) {
        return name.substr(name.size() - maxSignalNameLength);
    }

    return name;
}

template <typename Value = double>
struct Hist {
    template <typename T>
    Hist(std::vector<T> buckets, std::string namePrefix)
        : _buckets(std::move(bucketCast(buckets)))
        , _name(std::move(namePrefix))
        , _counters(_buckets.size(), 0)
    {
        if (std::adjacent_find(_buckets.begin(), _buckets.end(), std::greater_equal<Value>()) != _buckets.end()) {
            throw std::invalid_argument("Vector of buckets bounds is unsorted");
        }
        if (_buckets.empty()) {
            throw std::invalid_argument("Vector of buckets bounds is empty");
        }
    }

    Hist(std::vector<Value> buckets, std::string namePrefix)
        : _buckets(std::move(buckets))
        , _name(std::move(namePrefix))
        , _counters(_buckets.size(), 0)
    {
        if (std::adjacent_find(_buckets.begin(), _buckets.end(), std::greater_equal<Value>()) != _buckets.end()) {
            throw std::invalid_argument("Vector of buckets bounds is unsorted");
        }
        if (_buckets.empty()) {
            throw std::invalid_argument("Vector of buckets bounds is empty");
        }
    }

    void update(const Value& value) {
        const auto it = std::upper_bound(std::begin(_buckets), std::end(_buckets), value);
        if (it != std::begin(_buckets)) {
            _counters[std::distance(std::begin(_buckets), it) - 1] += 1;
        }
    }

    NamedHist<Value> get() const {
        std::vector<std::tuple<Value, std::size_t>> result;
        for (std::size_t i = 0; i < _counters.size(); ++i) {
            result.emplace_back(_buckets[i], _counters[i]);
        }
        result.emplace_back(_buckets.back() * 2, 0);
        return std::tuple{withSigoptSuffix(_name, "hgram"), result};
    }

private:
    template <typename T>
    static std::vector<Value> bucketCast(const std::vector<T>& buckets) {
        using boost::adaptors::transformed;

        const auto casted = buckets | transformed([] (const auto& v) {return Value(v);});

        return std::vector<Value>(std::begin(casted), std::end(casted));
    }

    const std::vector<Value> _buckets;
    const std::string _name;
    std::vector<std::size_t> _counters;
};


struct CountSubstring {
    explicit CountSubstring(std::string substring, std::string_view namePrefix = {})
        : _substring(std::move(substring))
        , _searcher(_substring.begin(), _substring.end())
        , _name(normalizeName(namePrefix.empty() ? _substring : buildSignalName(namePrefix,  _substring)))
        , _counter(0)
    {}

    void update(const std::map<std::string, std::string>& record) {
        static const std::string message = "message";
        static const std::string ecMessage = "error_code.message";
        if (foundIn(message, record) || foundIn(ecMessage, record)) {
            _counter += 1;
        }
    }

    NamedValue<std::size_t> get() const {
        return {withSigoptSuffix(_name, "summ"), _counter};
    }

private:
    bool foundIn(const std::string& field, const std::map<std::string, std::string>& record) const {
        if (const auto it = record.find(field); record.end() != it) {
            const auto& value = it->second;
            return value.end() != std::search(value.begin(), value.end(), _searcher);
        }
        return false;
    }

    const std::string _substring;
    const boost::algorithm::boyer_moore<std::string_view::iterator> _searcher;
    const std::string _name;
    std::size_t _counter;
};

using CountErrors = CountSubstring;

struct CountByHttpStatus {
    explicit CountByHttpStatus(std::string uriFilter = "", std::string_view namePrefix = "")
        : _uriFilter(std::move(uriFilter))
        , _namePrefix(normalizeName(buildSignalName(namePrefix, _uriFilter)))
        , _counters{}
    {}

    void update(std::size_t status) {
        if (status >= 0 && status < _counters.size()) {
            _counters[status] += 1;
        }
    }

    std::vector<NamedValue<std::size_t>> get() const {
        std::vector<NamedValue<std::size_t>> result;
        for (std::size_t i = 0; i < _counters.size(); ++i) {
            if (0 < _counters[i]) {
                result.emplace_back(withSigoptSuffix(_namePrefix + "_" + std::to_string(i), "summ"), _counters[i]);
            }
        }

        return result;
    }

private:
    static constexpr std::size_t maxHttpStatus = 600;
    const std::string _uriFilter;
    const std::string _namePrefix;
    std::array<std::size_t, maxHttpStatus> _counters;
};

} // namespace unistat
