#pragma once

#include <mail/unistat/cpp/include/meters/common.h>


namespace unistat::CountByCondition {

enum Condition {
    Equals = 0,
    Contains,
    AnyStr,
};


using ConditionPair = std::pair<Condition, std::string>;
using ConditionStrPair = std::pair<ConditionPair, std::string>;
using ConditionStrPairVec = std::vector<ConditionStrPair>;


inline bool match(Condition cond, std::string_view needle, std::string_view haystack) {
    switch(cond) {
        case Equals:
            return needle == haystack;
        case Contains:
            return haystack.find(needle) != haystack.npos;
        case AnyStr:
            return true;
        default:
            throw std::invalid_argument("unknown condition value");
    }
}


inline bool match(const ConditionPair& filter, std::string_view haystack) {
    return match(filter.first, filter.second, haystack);
}


struct Counter {
    explicit Counter(std::string key, ConditionStrPairVec filter2signal, ConditionStrPairVec baseFilters)
        : key_{std::move(key)}
        , filter2signal_{std::move(filter2signal)}
        , baseFilters_{std::move(baseFilters)} {
        
        if (key_.empty()) {
            throw std::invalid_argument("key is empty");
        }
        if (filter2signal_.empty()) {
            throw std::invalid_argument("filter2signal are empty");
        }

        signalValues_.reserve(filter2signal.size());
        for (const auto& [filter, signal] : filter2signal_) {
            if (signal.empty()) {
                throw std::invalid_argument("signal name is empty");
            }
            signalValues_.insert_or_assign(signal, 0U);
            match(filter, "sanity check");
        }

        for (const auto& [filter, logField] : baseFilters_) {
            if (logField.empty()) {
                throw std::invalid_argument("logField name is empty");
            }
            match(filter, "sanity check");
        }
    }

    void update(const std::map<std::string, std::string>& record) {
        for (const auto& [filter, logField] : baseFilters_) {
            const auto it = record.find(logField);
            if (record.end() == it || !match(filter, it->second)) {
                return;
            }
        }

        const auto it = record.find(key_);
        if (record.end() == it) {
            return;
        }

        const std::string& logValue = it->second;
        for (const auto& [filter, signal] : filter2signal_) {
            if (!match(filter, logValue)) {
                continue;
            }
            ++signalValues_[signal];
            break;
        }
    }

    std::vector<NamedValue<std::size_t>> get() const {
        std::vector<NamedValue<std::size_t>> result;
        result.reserve(signalValues_.size());
        for (const auto& [signal, value] : signalValues_) {
            result.emplace_back(signal, value);
        }
        return result;
    }
private:
    const std::string key_;
    const ConditionStrPairVec filter2signal_;
    const ConditionStrPairVec baseFilters_;

    std::unordered_map<std::string, std::size_t> signalValues_;
};


using CounterPtr = std::shared_ptr<Counter>;

}
