#pragma once

#include <mail/unistat/cpp/include/meters/common.h>

#include <map>
#include <vector>
#include <re2/re2.h>

namespace unistat {

struct SupervisorLogRestartMeters {
    explicit SupervisorLogRestartMeters(std::string namePrefix)
        : _regexUnexpectedRestart(
                R"raw([0-9\-]+ [0-9:,]+ [a-zA-Z]+ exited: ([a-z0-9A-Z_-]+) \(([^;]+); not expected\))raw"
        )
        , _regexExpectedRestart(R"raw([0-9\-]+ [0-9:,]+ [a-zA-Z]+ stopped: ([a-z0-9A-Z_-]+) \(([^;]+)\))raw")
        , _name(normalizeName(namePrefix))
        , _counters()
    {}

    void update(std::string_view record) {
        std::string serviceName;
        std::string description;
        const re2::StringPiece recordPiece(record.data(), record.size());
        if (re2::RE2::PartialMatch(recordPiece, _regexUnexpectedRestart, &serviceName, &description)) {
            const std::string normalizedPrefix = normalizeName(_name + "_" + serviceName + "_" + description);
            const std::string signalName = withSigoptSuffix(normalizedPrefix, "summ");
            _counters[signalName] += 1;
        } else if (re2::RE2::PartialMatch(recordPiece, _regexExpectedRestart, &serviceName, &description)) {
            const std::string normalizedPrefix = normalizeName(_name + "_" + serviceName + "_" + description);
            const std::string signalName = withSigoptSuffix(normalizedPrefix, "summ");
            _counters[signalName] += 1;
        }
    }

    std::vector<NamedValue<std::size_t>> get() const {
        std::vector<NamedValue<std::size_t>> result;
        result.reserve(_counters.size());
        for (const auto& p : _counters) {
            result.push_back(NamedValue<std::size_t>{p.first, p.second});
        }
        return result;
    }

private:
    const re2::RE2 _regexUnexpectedRestart;
    const re2::RE2 _regexExpectedRestart;
    const std::string _name;
    std::map<std::string, std::size_t> _counters;
};

} // namespace unistat
