#pragma once

#include <mail/unistat/cpp/include/meters/common.h>
#include <mail/unistat/cpp/include/signal_record.h>

namespace unistat {

struct HttpClientHttpRequestCountByStatus {
    explicit HttpClientHttpRequestCountByStatus(std::string uriFilter, std::string namePrefix)
        : _uriFilter(std::move(uriFilter))
        , _searcher(_uriFilter.begin(), _uriFilter.end())
        , _impl(_uriFilter, std::move(namePrefix))
    {}

    void update(const std::map<std::string, std::string>& record) {
        auto statusIt = record.find("status");

        if (record.end() == statusIt) {
            throw std::invalid_argument(std::string("Can't find value with key \"status\" in argument of ")
                                     + __PRETTY_FUNCTION__);
        }
        if (matchEndpoint(record)) {
            _impl.update(std::stoul(statusIt->second));
        }
    }

    decltype(auto) get() const {
        return _impl.get();
    }

    bool matchEndpoint(const std::map<std::string, std::string>& record) {
        auto uriIt = record.find("uri");
        if (record.end() == uriIt) {
            throw std::invalid_argument(std::string("Can't find value with key \"uri\" in argument of ")
                                     + __PRETTY_FUNCTION__);
        }
        return uriIt->second.end() != std::search(uriIt->second.begin(), uriIt->second.end(), _searcher);
    }

private:
    const std::string _uriFilter;
    const boost::algorithm::boyer_moore<std::string::const_iterator> _searcher;
    CountByHttpStatus _impl;
};

template <typename BucketBound = std::chrono::milliseconds, template<typename> typename H = Hist>
struct HttpClientHttpRequestTotalTimeHistBase {
    template <typename T>
    HttpClientHttpRequestTotalTimeHistBase(const std::vector<T>& buckets, std::string uriFilter, std::string namePrefix)
        : _uriFilter(std::move(uriFilter))
        , _impl(buckets, std::move(namePrefix))
    {}

    void update(const std::map<std::string, std::string>& record) {
        const auto totalTimeIt = record.find("total_time");
        if (record.end() == totalTimeIt) {
            throw std::invalid_argument(std::string("Can't find value with key \"total_time\" in argument of ")
                                     + __PRETTY_FUNCTION__);
        }

        const auto uriIt = record.find("uri");
        if (record.end() == uriIt) {
            throw std::invalid_argument(std::string("Can't find value with key \"uri\" in argument of ")
                                     + __PRETTY_FUNCTION__);
        }

        if (std::string::npos != uriIt->second.find(_uriFilter)) {
            const auto totalTime = std::chrono::duration_cast<BucketBound>(
                    std::chrono::duration<double>(std::stod(totalTimeIt->second))
            );
            _impl.update(totalTime);
        }
    }

    decltype(auto) get() const {
        return _impl.get();
    }

private:
    const std::string _uriFilter;
    H<BucketBound> _impl;
};

using HttpClientHttpRequestTotalTimeHist = HttpClientHttpRequestTotalTimeHistBase<>;

} // namespace unistat
