#pragma once

#include <mail/unistat/cpp/include/signal_record.h>
#include <mail/unistat/cpp/include/meters/common.h>
#include <mail/unistat/cpp/include/meters/request_matcher.h>

#include <boost/range/adaptor/transformed.hpp>
#include <boost/lexical_cast.hpp>

namespace unistat {

namespace detail {

template <RequestMatcher Matcher>
struct AccessLogCount {
    explicit AccessLogCount(const std::string& uriFilter = "", std::string_view namePrefix = "xxx")
        : _matcher(uriFilter)
        , _name(normalizeName(buildSignalName(namePrefix, uriFilter)))
        , _counter(0)
    {}

    void update(const std::map<std::string, std::string>& record) {
        const auto it = record.find("request");
        if (it != record.end() && _matcher.match(it->second)) {
            _counter += 1;
        }
    }

    NamedValue<std::size_t> get() const {
        return {withSigoptSuffix(_name, "summ"), _counter};
    }

private:
    Matcher _matcher;
    const std::string _name;
    std::size_t _counter;
};

template <RequestMatcher Matcher>
struct AccessLogCountByFirstStatusDigit {
    explicit AccessLogCountByFirstStatusDigit(const std::string& uriFilter = "", std::string_view namePrefix = "")
        : _matcher(uriFilter)
        , _namePrefix(normalizeName(buildSignalName(namePrefix, uriFilter)))
        , _counters{}
    {}

    void update(const std::map<std::string, std::string>& record) {
        using namespace std::literals;

        auto requestIt = record.find("request");

        if (record.end() == requestIt ) {
            throw std::invalid_argument("Can't find value with key \"request\" in argument of "s + __PRETTY_FUNCTION__);
        }

        if (_matcher.match(requestIt->second)) {
            auto statusIt = record.find("status_code");
            if (record.end() != statusIt) {
                long status;
                if (boost::conversion::try_lexical_convert(statusIt->second, status)) {
                    update(status);
                    return;
                } else {
                    throw std::invalid_argument("Can't convert value ('"s + statusIt->second + "') with key"
                            + " \"status_code\" to long in argument of " + __PRETTY_FUNCTION__);
                }
            } else {
                throw std::invalid_argument("Can't find value with key \"status_code\" in argument of "s
                        + __PRETTY_FUNCTION__);
            }
        }
    }

    void update(long status) {
        const long firstStatusDigit = status / 100;
        if (firstStatusDigit < 10  && firstStatusDigit >= 0) {
            _counters[firstStatusDigit] += 1;
        }
    }

    std::vector<NamedValue<std::size_t>> get() const {
        std::vector<NamedValue<std::size_t>> result;
        for (std::size_t i = 0; i < _counters.size(); ++i) {
            if (0 < _counters[i]) {
                result.emplace_back(withSigoptSuffix(_namePrefix + std::to_string(i) + "xx", "summ"), _counters[i]);
            }
        }
        return result;
    }

private:
    Matcher _matcher;
    const std::string _namePrefix;
    std::array<std::size_t, 10> _counters;
};

template <RequestMatcher Matcher, typename BucketBound = std::chrono::milliseconds, template<typename> typename H = Hist>
struct AccessLogRequestTimeHistBase {
    template <typename T>
    explicit AccessLogRequestTimeHistBase(
        std::vector<T> buckets
        , const std::string& uriFilter = ""
        , std::string_view namePrefix = "access_log_request"
    )
        : _matcher(uriFilter)
        , _impl(
            std::move(buckets)
            , normalizeName(buildSignalName(namePrefix, uriFilter))
        )
    {}

    void update(const std::map<std::string, std::string>& record) {
        using namespace std::literals;

        auto requestIt = record.find("request");

        if (record.end() == requestIt ) {
            throw std::invalid_argument("Can't find value with key \"request\" in argument of "s + __PRETTY_FUNCTION__);
        }

        if (_matcher.match(requestIt->second)) {
            auto timeIt = record.find("profiler_exec");
            if (record.end() != timeIt) {
                double time;
                if (boost::conversion::try_lexical_convert(timeIt->second, time)) {
                    _impl.update(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::duration<double>(time)));
                } else {
                    throw std::invalid_argument("Can't convert value ('"s + timeIt->second + "') with key"
                            + " \"profiler_exec\" to double in argument of " + __PRETTY_FUNCTION__);
                }
            } else {
                throw std::invalid_argument("Can't find value with key \"profiler_exec\" in argument of "s
                        + __PRETTY_FUNCTION__);
            }
        }
    }

    auto get() const {
        return _impl.get();
    }

private:
    Matcher _matcher;
    H<BucketBound> _impl;
};

}

struct AccessLogCount : detail::AccessLogCount<SubstringMatcher> {
    AccessLogCount(std::string_view namePrefix = "")
        : detail::AccessLogCount<SubstringMatcher>("" /* uriFilter */, namePrefix)
    {}
};

struct AccessLogCountByFirstStatusDigit : detail::AccessLogCountByFirstStatusDigit<SubstringMatcher> {
    AccessLogCountByFirstStatusDigit(std::string_view namePrefix = "")
        : detail::AccessLogCountByFirstStatusDigit<SubstringMatcher>("" /* uriFilter */, namePrefix)
    {}
};

struct AccessLogRequestTimeHist : detail::AccessLogRequestTimeHistBase<SubstringMatcher> {
    template <typename T>
    AccessLogRequestTimeHist(std::vector<T> buckets, std::string_view namePrefix = "")
        : detail::AccessLogRequestTimeHistBase<SubstringMatcher>(std::move(buckets), "" /* uriFilter */, namePrefix)
    {}
};

using AccessLogCountByPath = detail::AccessLogCount<PathMatcher>;
using AccessLogCountByPathAndFirstStatusDigit = detail::AccessLogCountByFirstStatusDigit<PathMatcher>;
using AccessLogRequestTimeHistByPath = detail::AccessLogRequestTimeHistBase<PathMatcher>;

} // namespace unistat
