#pragma once

#include <boost/asio.hpp>
#include <boost/filesystem.hpp>

#include <sys/stat.h>

namespace unistat {

namespace filesystem = boost::filesystem;

struct File {
    explicit File(boost::asio::io_service& ios, std::string_view pathfile)
        : _path(pathfile)
        , _fd(ios)
        , _inode()
        , _alignment(0)
    {}

    //! Default value is false because reopen occurs after log rotation usually
    void reopen(bool fastForward = false) {
        close();
        open(fastForward);
    }

    std::size_t alignment() const {
        return _alignment;
    }

    void open(bool fastForward = true);

    void close() {
        _fd.close();
    }

    bool isOpen() const {
        return _fd.is_open();
    }

    const std::string& getPath() const {
        return _path;
    }

    bool isRenamed() const {
        const struct stat statBuf = getStat();
        return statBuf.st_ino != _inode;
    }

    bool isFileExist() const {
        boost::system::error_code ec;
        bool isExists = filesystem::exists(_path, ec);
        return !ec && isExists;
    }

    bool shouldReopenFile() const {
        return isFileExist() && isRenamed();
    }

    struct stat getStat() const;


    template <typename MutableBufferSequence, typename CompletionCondition>
    std::size_t read(
        const MutableBufferSequence& buffers
        , CompletionCondition completionCondition
        , boost::system::error_code& ec
    ) {
        return boost::asio::read(_fd, buffers, completionCondition, ec);
    }

private:
    std::string _path;
    boost::asio::posix::stream_descriptor _fd;
    std::streampos _offset;
    ino_t _inode;
    std::size_t _alignment;
};

} // namespace unistat
