#pragma once

#include <type_traits>

#include <mail/unistat/cpp/include/readers/file.h>
#include <mail/unistat/cpp/include/readers/text_buffer.h>

#include <mail/unistat/cpp/include/logger.h>
#include <boost/asio.hpp>

namespace unistat {

template <typename File = File, typename Buffer = TextBuffer, typename Logger = Logger>
struct TextFileReader {
    explicit TextFileReader(
        File file
        , bool fastForward = true
        , Buffer buffer = Buffer()
    )
        : _file(std::move(file))
        , _buffer(std::move(buffer))
        , _logger()
        , _sleepTime(50)
    {
        _file.open(fastForward);
    }

    std::string_view operator()() {
        std::optional<std::string_view> logLine;

        while (!logLine) {
            boost::system::error_code ec;
            try {
                logLine = tryGetLine(ec);
            } catch(const std::exception& e) {
                LOGDOG_WHERE_(logger(), error, message="Error during tryGetLine", exception=e);
                _buffer.reset();
            }

            if (logLine) {
                return *logLine;
            }

            if (ec) {
                try {
                    if (_file.shouldReopenFile()) {
                        LOGDOG_WHERE_(logger(), debug, message="Reopen");
                        reopen();
                    } else {
                        adaptedSleep();
                    }
                } catch (const std::exception &e) {
                    LOGDOG_WHERE_(logger(), error, message="Unexpected exception", exception=e);
                    _buffer.reset();
                }
            }
        }

        return *logLine;
    }

    std::string getSourceName() const {
        return _file.getPath();
    }

    void setLogger(Logger logger) {
        if constexpr (std::is_same<Logger, logdog::none_t>::value) {
            _logger = makeLogger(std::move(logger), "");
        } else {
            _logger = makeLogger(std::move(logger), getSourceName());
        }
    }


private:
    void adaptedSleep() {
        using namespace std::chrono;

        const auto now = steady_clock::now();
        const milliseconds fromLastSleep = duration_cast<milliseconds>(now - _lastSleep);
        if (fromLastSleep < 2 * _sleepTime && _sleepTime < 1000ms) {
            _sleepTime *= 2;
        } else if (fromLastSleep > 10 * _sleepTime) {
            _sleepTime = milliseconds(50);
        } else if (fromLastSleep > 2 * _sleepTime && _sleepTime >= 100ms) {
            _sleepTime /= 2;
        }
        _lastSleep = now;
        LOGDOG_WHERE_(logger(), debug, message=std::string("Buffer is empty. Sleep for ")
                + std::to_string(_sleepTime.count()) + "ms");
        std::this_thread::sleep_for(_sleepTime);
    }

    void reopen() {
        _file.reopen();
        _buffer.reset();
    }

    std::optional<std::string_view> tryGetLine(boost::system::error_code& ec) {
        std::optional<std::string_view> lineOpt = _buffer.tryGetLine(_file, ec);
        if (ec) {
            if (boost::asio::error::eof == ec.value()) {
                LOGDOG_WHERE_(logger(), debug, message="EOF");
            } else {
                LOGDOG_WHERE_(logger(), error, error_code=ec);
            }
        }

        if (!_file.isOpen()) {
            LOGDOG_WHERE_(logger(), warning, message="File is not opened");
        }

        return lineOpt;
    }

    static auto makeLogger(Logger logger, std::string str) {
        if constexpr (std::is_same<Logger, logdog::none_t>::value) {
            return logger;
        } else {
            return logdog::bind(std::move(logger), source_name=str);
        }
    }

    using TextFileReaderLogger = decltype(TextFileReader::makeLogger(std::declval<Logger>(), ""));

    TextFileReaderLogger logger() {
        if (!_logger) {
            throw std::logic_error("Trying use uninitialized logger");
        }

        return *_logger;
    }

    File _file;
    Buffer _buffer;
    std::optional<TextFileReaderLogger> _logger;
    std::chrono::steady_clock::time_point _lastSleep;
    std::chrono::milliseconds _sleepTime;
};

template <typename File = File, typename Buffer = PaBuffer, typename Logger = Logger>
using PaReader = TextFileReader<File, Buffer, Logger>;

} //namespace unistat
