#pragma once

#include <optional>
#include <memory>

#include <boost/asio.hpp>

#include <pa/interface.h>

namespace unistat {

struct GetPaRecord {
    static constexpr off_t wpe_size = static_cast<off_t>(sizeof(pa::wmi_profiler_entry));
    static constexpr std::size_t skipSize = 0;

    static std::optional <std::string_view> get(std::string_view buffer) {
        if (buffer.size() >= wpe_size) {
            return std::string_view(buffer.data(), wpe_size);
        }
        return std::nullopt;
    }
};

struct GetLine {
    static constexpr std::size_t skipSize = 1;

    static std::optional <std::string_view> get(std::string_view buffer) {
        const auto newLinePos = buffer.find('\n');
        if (newLinePos != std::string_view::npos) {
            return std::string_view(buffer.data(), newLinePos);
        }
        return std::nullopt;
    }
};

template <typename GetRecord>
struct Buffer {
    explicit Buffer(std::size_t size = DEFAULT_BUFFER_SIZE)
        : _buffer(new char[size])
        , _startNewLine(0)
        , _size(0)
        , _capacity(size)
        , _nextCapacity(_capacity)
        , _lastGetLineFail(false)
    {}

    bool needReadMore() const noexcept {
        return empty() || _lastGetLineFail;
    }

    std::size_t size() const noexcept {
        return _size;
    }

    std::size_t capacity() const noexcept {
        return _capacity;
    }

    bool empty() const noexcept {
        return 0 == _size;
    }

    void reset() {
        _size = 0;
        _startNewLine = 0;
    }

    template <typename Source>
    std::optional<std::string_view> tryGetLine(Source& source, boost::system::error_code& ec) {
        if (needReadMore()) {
            std::size_t readSize = read(source, ec);
            if (!readSize) {
                return std::nullopt;
            }
        }

        const std::string_view buffer_view(_buffer.get() + _startNewLine, _size - _startNewLine);

        std::optional<std::string_view> result = GetRecord::get(buffer_view);
        if (result) {
            _startNewLine += result->size() + GetRecord::skipSize;
            _lastGetLineFail = false;
            return result;
        }

        _lastGetLineFail = true;

        if (MAX_BUFFER_SIZE < _size * 2 && 0 == _startNewLine) {
            throw std::runtime_error("Very long line. MAX_BUFFER_SIZE = " + std::to_string(MAX_BUFFER_SIZE)
                    + ". Length of line is at least " + std::to_string(_size));
        }

        if (_capacity != _nextCapacity && _nextCapacity > buffer_view.size()) {
            _capacity = _nextCapacity;
            std::unique_ptr<char[]> newBuffer(new char[_capacity]);
            std::copy(buffer_view.begin(), buffer_view.end(), newBuffer.get());
            _buffer = std::move(newBuffer);
        } else {
            std::copy(buffer_view.begin(), buffer_view.end(), _buffer.get());
        }
        _size -= _startNewLine;
        _startNewLine = 0;

        return std::nullopt;
    }

    template <typename Source>
    std::size_t read(Source& source, boost::system::error_code& ec) {
        using boost::asio::transfer_at_least;
        const std::size_t readSize = source.read(bufferForRead(), transfer_at_least(atLeastReadSize()), ec);

        updateSize(readSize);

        return readSize;
    }

    static constexpr std::size_t DEFAULT_BUFFER_SIZE = 4096;
    static constexpr std::size_t MIN_BUFFER_SIZE = 256;
    static constexpr std::size_t MAX_BUFFER_SIZE = 1024 * 1024;
private:
    boost::asio::mutable_buffer bufferForRead() {
        return boost::asio::buffer(_buffer.get() + _size, _capacity - _size);
    }

    void adaptSize(const std::size_t readSize) {
        if (readSize + _size < _capacity * 0.9) {
            _nextCapacity = std::max(MIN_BUFFER_SIZE, _capacity / 2);
        } else if (readSize + _size > _capacity * 0.95) {
            _nextCapacity = std::min(MAX_BUFFER_SIZE, _capacity * 2);
        }
    }

    void updateSize(std::size_t readSize) {
        adaptSize(readSize);
        _size += readSize;
        _lastGetLineFail = false;
    }

    std::size_t atLeastReadSize() const {
        return _capacity - _size;
    }

    std::unique_ptr<char[]> _buffer;
    std::size_t _startNewLine;
    std::size_t _size;
    std::size_t _capacity;
    std::size_t _nextCapacity;
    bool _lastGetLineFail;
};

using TextBuffer = Buffer<GetLine>;
using PaBuffer = Buffer<GetPaRecord>;

} // namespace unistat
