#pragma once

#include <mail/unistat/cpp/include/logger.h>

#include <vector>

#include <ymod_webserver/server.h>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/filtered.hpp>

namespace unistat {

class RunStatus final {
public:
    operator bool() const noexcept {
        return running.load(std::memory_order_acquire);
    }

    void reset() noexcept {
        running.store(false, std::memory_order_release);
    }

private:
    std::atomic<bool> running{ true };
};

template <typename Logs>
struct Context {
    explicit Context(std::vector<Logs> logs)
        : _logs(std::move(logs))
        , _isStop()
    {}

    std::string getSignals() const {
        std::vector<std::string> signals;
        for (const auto& l : _logs) {
            auto stat = std::visit([](const auto& l){return l->get();}, l);
            signals.insert(signals.end(), std::move_iterator(stat.begin()), std::move_iterator(stat.end()));
        }

        using boost::algorithm::join;
        using boost::adaptors::filtered;

        const auto notEmptySignals = signals | filtered([] (const std::string& s) {return !s.empty();});
        const std::string serializedSignals = join(notEmptySignals, ",");

        std::stringstream out;
        out << "[" << serializedSignals << "]";

        return out.str();
    }

    std::vector<Logs> _logs;
    RunStatus _isStop;
};

inline void initYplatformGlobalLogger(std::string path, std::string logLevel) {
    yplatform::log::log_settings logSettings;
    logSettings.async = true;
    logSettings.queue_size = 1048576;
    logSettings.level = logLevel;
    logSettings.format = "%v";

    yplatform::log::sink_settings sink;
    sink.type = yplatform::log::sink_type::file;
    sink.force_flush = true;
    sink.path = std::move(path);

    logSettings.sinks.emplace_back(std::move(sink));


    yplatform::log::settings settings;
    settings.logs.emplace("global", std::move(logSettings));
    yplatform::log::global_log_init(settings);
}

template <typename Logs>
void startWebServer(
        boost::asio::io_service& io
        , const char * host
        , unsigned short port
        , std::shared_ptr<Context<Logs>> logContext
        , yplatform::log::source logger
) {
    ymod_webserver::settings settings;
    settings.endpoints.emplace("", ymod_webserver::endpoint("", host, port));
    settings.access_log = std::move(logger);
    ymod_webserver::server server(io, settings);

    server.bind("", {"/ping"}, [](ymod_webserver::http::stream_ptr stream) {
        stream->set_code(ymod_webserver::codes::ok);
        stream->result_body("pong");
    });
    server.bind("", {"/woof"}, [](ymod_webserver::http::stream_ptr stream) {
        stream->set_code(ymod_webserver::codes::ok);
        stream->result_body("WOOF");
    });
    server.bind("", {"/unistat"}, [logContext](ymod_webserver::http::stream_ptr stream) {
        stream->set_code(ymod_webserver::codes::ok);
        stream->result_body(logContext->getSignals());
    });
    server.start();
    io.run();
}

template <typename Logs>
void run(
        const char* host
        , unsigned short port
        , std::vector<Logs> logs
        , const char* yplatformLogPath = "yplatform.log"
        , const char* logLevel = "info"
) {
    if (yplatformLogPath && *yplatformLogPath) {
        initYplatformGlobalLogger(yplatformLogPath, logLevel);        
    }
    auto context = std::make_shared<Context<Logs>>(std::move(logs));

    auto logger = getLogger();

    auto logWorker = [] (auto context, auto log) {
        while (context->_isStop) {
            std::visit([](auto& l){
                l->parseAndUpdateMeters(l->read());
            }, log);
        }
    };
    std::vector<std::thread> threads;
    for (const auto& l : context->_logs) {
        std::visit([&logger](auto&l){l->setLogger(logger);}, l);
        threads.emplace_back(logWorker, context, l);
    }

    boost::asio::io_service io;
    startWebServer(io, host, port, context, YGLOBAL_LOG_SERVICE);
    for (auto& t : threads) {
        t.join();
    }
}

} // namespace unistat
