#!/usr/bin/env python2
# coding: utf-8

import ctypes
import logging
import multiprocessing
import os
import re
import yaml

from argparse import ArgumentParser
from collections import namedtuple
from datetime import timedelta
from unistat.run import run, Log
from unistat.readers import (
    CoreDumpChecker,
    PaLogReader,
    StatServerXmlReader,
    TextFileReader,
)
from unistat.parsers import (
    identity,
    parse_ymod_httpclient_log_event,
    parse_ymod_webserver_access_log_record,
)
from unistat.meters import (
    AccessLogCount,
    AccessLogCountByFirstStatusDigit,
    AccessLogCountByStatus,
    AccessLogRequestTimeHist,
    CoreDumpCountByNameAndSignal,
    Hist,
    StatServerXmlReactorsTraceMax,
    YplatformLogHttpRequestBytesInHist,
    YplatformLogHttpRequestCountByStatus,
    YplatformLogHttpRequestTotalTimeHist,
    with_sigopt_suffix,
)

logging.basicConfig(level=logging.INFO, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)
    with open(args.retriever_config) as f:
        retriever_config = make_retriever_config(yaml.load(f))
    run(
        host=args.host,
        port=args.port,
        interval=args.interval,
        logs=(
            Log(
                path=retriever_config.access_log,
                reader=TextFileReader,
                parse_record=parse_ymod_webserver_access_log_record,
                meters=(
                    AccessLogCount(),
                    AccessLogCountByStatus(),
                    AccessLogCountByFirstStatusDigit(),
                    AccessLogRequestTimeHist(buckets=(0, 0.1, 0.3, 1.0, 5.0, 10.0)),
                )
            ),
            Log(
                path=retriever_config.yplatform_log,
                reader=TextFileReader,
                parse_record=parse_ymod_httpclient_log_event,
                meters=(
                    YplatformLogHttpRequestCountByStatus(
                        name_prefix='storage_http_status',
                        uri_filter=lambda w: 'storage.' in w,
                    ),
                    YplatformLogHttpRequestTotalTimeHist(
                        name='storage_http_timings',
                        buckets=(0, 0.1, 0.3, 1.0, 5.0, 10.0),
                        uri_filter=lambda w: 'storage.' in w,
                    ),
                    YplatformLogHttpRequestBytesInHist(
                        name='storage_bytes_in',
                        buckets=(0, 1024, 4 * 1024, 256 * 1024, 1024 ** 2, 4 * 1024 ** 2, 15 * 1024 ** 2),
                        uri_filter=lambda w: 'storage.' in w,
                    ),
                    YplatformLogHttpRequestCountByStatus(
                        name_prefix='resize_genurl_http_status',
                        uri_filter=lambda w: 'resize.' in w and '/genurl' in w,
                    ),
                    YplatformLogHttpRequestTotalTimeHist(
                        name='resize_genurl_http_timings',
                        buckets=(0, 0.1, 0.2, 0.5, 1.0),
                        uri_filter=lambda w: 'resize.' in w and '/genurl' in w,
                    ),
                    YplatformLogHttpRequestCountByStatus(
                        name_prefix='resize_get_http_status',
                        uri_filter=lambda w: 'resize.' in w and '/genurl' not in w,
                    ),
                    YplatformLogHttpRequestTotalTimeHist(
                        name='resize_get_http_timings',
                        buckets=(0, 0.1, 0.3, 1.0, 5.0, 10.0),
                        uri_filter=lambda w: 'resize.' in w and '/genurl' not in w,
                    ),
                    YplatformLogHttpRequestBytesInHist(
                        name='resize_get_bytes_in',
                        buckets=(0, 1024, 4 * 1024, 256 * 1024, 1024 ** 2, 4 * 1024 ** 2, 15 * 1024 ** 2),
                        uri_filter=lambda w: 'resize.' in w and '/genurl' not in w,
                    ),
                    YplatformLogHttpRequestCountByStatus(
                        name_prefix='hound_http_status',
                        uri_filter=lambda w: '/mimes' in w,
                    ),
                    YplatformLogHttpRequestTotalTimeHist(
                        name='hound_http_timings',
                        buckets=(0, 0.1, 0.3, 1.0, 3.0),
                        uri_filter=lambda w: '/mimes' in w,
                    ),
                ),
            ),
            Log(
                path=retriever_config.profiler_log,
                reader=PaLogReader,
                parse_record=identity,
                meters=(
                    StorageRequestTimeHists(),
                ),
            ),
            Log(
                path=retriever_config.retriever_log,
                reader=TextFileReader,
                parse_record=identity,
                meters=(
                    CountPartIds(),
                    CountErrors(),
                ),
            ),
            Log(
                path=retriever_config.stat_server,
                reader=StatServerXmlReader,
                parse_record=identity,
                meters=(
                    StatServerXmlReactorsTraceMax(),
                ),
            ),
            Log(
                path='var/cores',
                reader=CoreDumpChecker,
                parse_record=identity,
                meters=(
                    CoreDumpCountByNameAndSignal(),
                ),
            ),
        )
    )


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-i', '--interval', default=timedelta(seconds=5),
                        type=lambda v: timedelta(seconds=int(v)))
    parser.add_argument('retriever_config')
    return parser.parse_args()


def make_retriever_config(data):
    return RetrieverConfig(
        yplatform_log=os.path.join(os.curdir, data['config']['log']['global']['sinks'][0]['path']),
        access_log=os.path.join(os.curdir, data['config']['log']['access']['sinks'][0]['path']),
        retriever_log=os.path.join(os.curdir, data['config']['log']['retriever']['sinks'][0]['path']),
        profiler_log=os.path.join(os.curdir, next(v['configuration']['profiler_log_name']
                                                  for v in data['config']['modules']['module']
                                                  if v['system']['name'] == 'retriever')),
        stat_server='localhost: %s ' % next(v['configuration']['endpoints']['listen']['_port']
                                            for v in data['config']['modules']['module']
                                            if v['system']['name'] == 'stat_server'),
    )


RetrieverConfig = namedtuple('RetrieverConfig', (
    'yplatform_log',
    'access_log',
    'retriever_log',
    'profiler_log',
    'stat_server',
))


class StorageRequestTimeHists(object):
    def __init__(self):
        self.__range_get_xml = Hist(
            name='storage_range_get_xml',
            buckets=(0, 0.1, 0.3, 1.0, 5.0, 10.0),
            get_value=lambda v: float(v.spent_ms) / 1000.0,
        )
        self.__get_range = Hist(
            name='storage_get_range',
            buckets=(0, 0.1, 0.3, 1.0, 5.0, 10.0),
            get_value=lambda v: float(v.spent_ms) / 1000.0,
        )

    def update(self, record):
        if record.req == 'range_get_xml':
            self.__range_get_xml.update(record)
        elif record.req == 'get_range':
            self.__get_range.update(record)

    def get(self):
        return dict(
            range_get_xml=self.__range_get_xml.get(),
            get_range=self.__get_range.get(),
        )


class CountPartIds(object):
    def __init__(self):
        self.__counters = multiprocessing.Array(ctypes.c_size_t, len(PART_ID_TYPES))

    def update(self, record):
        part_id_type = get_part_id_type(record)
        if part_id_type is not None:
            with self.__counters.get_lock():
                self.__counters.get_obj()[part_id_type] += 1

    def get(self):
        with self.__counters.get_lock():
            return {k: [with_sigopt_suffix('part_ids_%s' % PART_ID_TYPES[k], 'summ'), v]
                    for k, v in enumerate(self.__counters) if v}


def get_part_id_type(line):
    if 'part id is single message part' in line:
        return 0
    if 'part id is multiple message part' in line:
        return 1
    if 'part id is temporary' in line:
        return 2
    if 'part id is old' in line:
        return 3


PART_ID_TYPES = (
    'single_message_part',
    'multiple_message_part',
    'temporary',
    'old',
)


class CountErrors(object):
    def __init__(self):
        self.__counters = multiprocessing.Manager().dict()

    def update(self, record):
        error = get_error(record)
        if error is not None:
            if error not in self.__counters:
                self.__counters[error] = 1
            else:
                self.__counters[error] += 1

    def get(self):
        return {k: [with_sigopt_suffix('errors_%s' % k.replace(' ', ''), 'ammm'), v]
                for k, v in self.__counters.copy().iteritems()}


def get_error(line):
    match = ERROR_RE.search(line)
    return match.group(1).strip() if match else None


ERROR_RE = re.compile(r'status=error reason=\[([^\d=]*)')

main()
