# coding: utf-8

import datetime
import logging
import os
import signal
import time

from .log import log_lock

log = logging.getLogger('unistat')


def make_run_cache_updater(source, stop, reader, meters, parse_record):
    def impl():
        signal.signal(signal.SIGHUP, ignore_signal)
        signal.signal(signal.SIGINT, ignore_signal)
        signal.signal(signal.SIGTERM, ignore_signal)
        read_count = 0
        processed_count = 0
        try:
            with log_lock:
                log.info('start collecting records from %s in process with pid %s', source, os.getpid())
            after_exception = False
            while not stop.is_set():
                prev_count_print = datetime.datetime.now()
                try:
                    if after_exception:
                        time.sleep(3)
                    with reader:
                        while not stop.is_set():
                            for value in reader(stop):
                                read_count += 1
                                now = datetime.datetime.now()
                                if now - prev_count_print >= datetime.timedelta(seconds=5):
                                    if read_count:
                                        with log_lock:
                                            log.info('%s: %s r/s (%s p/s)', source, read_count / 5.0,
                                                     processed_count / 5.0)
                                        read_count = 0
                                        processed_count = 0
                                    prev_count_print = now
                                if value is None:
                                    continue
                                try:
                                    parsed = parse_record(value)
                                    if parsed is None:
                                        continue
                                    for meter in meters:
                                        try:
                                            meter.update(parsed)
                                        except Exception as e:
                                            with log_lock:
                                                log.exception(
                                                    'failed to update meter %s with parsed record %s '
                                                    + 'from value "%s": %s', meter, parsed, value, e
                                                )
                                except Exception as e:
                                    with log_lock:
                                        log.exception('failed to parse by %s value "%s": %s',
                                                      parse_record, value, e)
                                processed_count += 1
                                if stop.is_set():
                                    break
                except BaseException as e:
                    with log_lock:
                        log.exception(e)
                    after_exception = True
        finally:
            with log_lock:
                log.info('finish collecting records from %s', source)
    return impl


def ignore_signal(signum, frame):
    with log_lock:
        log.info('ignore %s signal in %s', signum, os.getpid())
