# coding: utf-8

import logging
import signal
import os

from collections import namedtuple
from datetime import timedelta
from multiprocessing import Process, Event as ProcessEvent
from werkzeug.serving import run_simple
from .api import Api
from .cache import make_run_cache_updater, ignore_signal
from .log import log_lock


Log = namedtuple('Log', ('path', 'reader', 'parse_record', 'meters'))
log = logging.getLogger('unistat')


def run(logs, interval=timedelta(seconds=5), host='::', port=26010):
    with Context(logs=logs, interval=interval) as context:
        run_simple(
            hostname=host,
            port=port,
            application=Api(meters=context.meters),
        )


class Context(object):
    readers = None
    meters = None
    __stop_process = None
    __processes = None

    def __init__(self, logs, interval):
        self.__logs = logs
        self.__interval = interval

    def __enter__(self):
        signal.signal(signal.SIGHUP, ignore_signal)
        signal.signal(signal.SIGTERM, ignore_signal)
        with log_lock:
            log.info('start unistat in process with pid %s', os.getpid())
            log.info('start collect data with interval %s', self.__interval)
        self.__stop_process = ProcessEvent()
        readers = [(v.path, v.reader(v.path), v.parse_record, v.meters) for v in self.__logs]
        self.meters = [v for l in self.__logs for v in l.meters]
        self.__processes = [
            Process(
                name='%s reader process' % path,
                target=make_run_cache_updater(
                    source=path,
                    stop=self.__stop_process,
                    reader=reader,
                    meters=meters,
                    parse_record=parse_record,
                ),
            ) for path, reader, parse_record, meters in readers
        ]
        self.readers = [v[1] for v in readers]
        for process in self.__processes:
            process.start()
        return self

    def __exit__(self, _exc_type, _exc_val, _exc_tb):
        with log_lock:
            log.info('finish collect data')
        self.__stop_process.set()
        for process in self.__processes:
            with log_lock:
                log.info('waiting for %s', process.name)
            process.join()
        signal.signal(signal.SIGHUP, signal.SIG_DFL)
        signal.signal(signal.SIGTERM, signal.SIG_DFL)
