#ifndef USER_JOURNAL_ASYNC_WRITER_H
#define USER_JOURNAL_ASYNC_WRITER_H

#include <user_journal/writer.h>
#include <user_journal/entry_ostream.h>
#include <user_journal/logging.h>
#include <internal/profiling.h>
#include <internal/mutations_queue.h>
#include <internal/misc.h>
#include <internal/write_helpers.h>
#include <internal/access_traits.h>

namespace user_journal {

class AsyncWriter : public Writer, boost::noncopyable {
public:
    AsyncWriter(MutationsQueuePtr queue, AccessTraits accessTraits,
            std::locale locale)
        : queue(queue), accessTraits(accessTraits), locale(locale) {
    }

    virtual void write(const std::string & uid, const Entry & v) const override {
        profiling::call([&]{doWrite(uid, v);}, profiler(), "AsyncWriter::write", uid);
    }

private:
    void doWrite(const std::string & uid, const Entry & v) const {
        std::ostringstream s;
        s.imbue(locale);
        auto out = ytskv::utf_value(ytskv::with_endl)(s);
        out << ytskv::attr("uid", key(uid))
            << ytskv::attr("tskv_format", accessTraits.tskvFormat)
            << ytskv::attr("tableName", accessTraits.tableName);
        const EntryMapper<decltype(out)> mapper(out);
        v.map(mapper);
        out << ytskv::endl;

        if( !queue->add(s.str()) ) {
            handleQueueOverflow(uid, v);
        }
    }

    void handleQueueOverflow(const std::string & uid, const Entry & v) const {
        std::ostringstream s;
        s << "write queue overflow, entry={" << v << "}";
        if(logger().get()) {
            logger()->error(uid, "AsyncWriter::write", s.str(), 0);
        } else {
            throw std::overflow_error(s.str());
        }
    }

    MutationsQueuePtr queue;
    AccessTraits accessTraits;
    std::locale locale;
    profiling::LogPtr profiler() const { return accessTraits.profiler;}
    logging::LogPtr logger() const { return accessTraits.logger; }
};

} // namespace user_journal

#endif /* USER_JOURNAL_ASYNC_WRITER_H */
