#ifndef PROFILING_H_152107122013
#define PROFILING_H_152107122013

#include <user_journal/profiling.h>
#include <user_journal/date.h>
#include <boost/type_traits.hpp>

namespace user_journal {
namespace profiling {
namespace details {

inline void write(LogPtr log, const Time & start, const std::string & operation,
        const std::string & info ) {
    const Time stop(currentDate());
    const Duration duration(stop.time_since_epoch() - start.time_since_epoch());
    log->write( operation, info, duration );
}

typedef boost::true_type True;
typedef boost::false_type False;

template <typename T>
auto resultIsVoid( const T & func) {
    return boost::is_void<decltype(func())>();
}

template <typename T>
void callWithProfiling( T func, LogPtr log, const std::string & operation,
        const std::string & info, const Time & start, const True & ) {
    func();
    write( log, start, operation, info );
}

template <typename T>
auto callWithProfiling( T func, LogPtr log, const std::string & operation,
        const std::string & info, const Time & start, const False & ) {
    auto retval = func();
    write( log, start, operation, info );
    return retval;
}


template <typename T>
auto callWithProfiling( T func, LogPtr log,
        const std::string & operation, const std::string & info ) {
    const Time start(currentDate());
    try {
        return callWithProfiling( func, log, operation, info, start, resultIsVoid(func) );
    } catch ( const std::exception & e ) {
        write( log, start, operation, info + " - execution interrupted with exception: " + e.what() );
        throw;
    } catch (...) {
        write( log, start, operation, info + " - execution interrupted with unknown exception" );
        throw;
    }
}

} // namespace details

template <typename T>
auto call( T func, LogPtr log,
        const std::string & operation, const std::string & info = "" ) {
    return log.get() ? details::callWithProfiling( func, log, operation, info ) : func();
}

} // namespace profiling
} // namespace user_journal


#endif /* PROFILING_H_152107122013 */
