#ifndef USER_JOURNAL_SERVICE_FACTORY_H
#define USER_JOURNAL_SERVICE_FACTORY_H

#include <user_journal/service.h>
#include <user_journal/profiling.h>
#include <user_journal/logging.h>
#include <user_journal/exception_handler.h>
#include <user_journal/connection_info.h>
#include <string>
#include <boost/property_tree/ptree.hpp>

namespace user_journal {

class ServiceFactory {
public:

#define __CONN_ATTRIBUTE( type, name ) \
    ServiceFactory & name( const type & v ) { c.name = v; return *this; }

    __CONN_ATTRIBUTE(std::string, tableName);
    __CONN_ATTRIBUTE(std::string, tskvFormat);
    __CONN_ATTRIBUTE(std::size_t, writeChunkSize);
    __CONN_ATTRIBUTE(Milliseconds, writePollTimeout);
    __CONN_ATTRIBUTE(ExceptionHandlerPtr, onSenderThreadException);
    __CONN_ATTRIBUTE(profiling::LogPtr, profiler);
    __CONN_ATTRIBUTE(logging::LogPtr, logger);
    __CONN_ATTRIBUTE(FileWriter, fileWriter);
    __CONN_ATTRIBUTE(std::locale, locale);

#undef __CONN_ATTRIBUTE

    ServiceFactory & tskv(const boost::property_tree::ptree & p, FileWriter writer) {
        fileWriter(writer);
        tableName(p.get<std::string>("table_name"));
        tskvFormat(p.get<std::string>("tskv_format"));
        writeChunkSize(p.get<std::size_t>("write_chunk_size"));
        writePollTimeout(Milliseconds(p.get<int>("write_poll_timeout_ms")));
        locale(std::locale(p.get<std::string>("locale", "").c_str()));
        return *this;
    }

    ServicePtr product() const {
        if( !c.fileWriter ) {
            throw std::invalid_argument("Bad file writer!!!");
        }
        if( c.tableName.empty() ) {
            throw std::invalid_argument("Bad tableName!!!");
        }
        if( c.tskvFormat.empty() ) {
            throw std::invalid_argument("Bad tskvFormat!!!");
        }
        AsyncConnectionInfo info(c);
        info.onSenderThreadException = getExceptionHandler(c);
        return create(c);
    }
private:
    ServicePtr create(const AsyncConnectionInfo & c) const;
    ExceptionHandlerPtr getExceptionHandler(const AsyncConnectionInfo & c) const {
        if( c.onSenderThreadException != defaultExceptionHandler) {
            return c.onSenderThreadException;
        }

        if( c.logger!=noLogging ) {
            return logSenderException(c.logger);
        }

        return cerrExceptionHandler;
    }
    AsyncConnectionInfo c;
};

} // namespace user_journal

#endif /* USER_JOURNAL_SERVICE_FACTORY_H */
