package ru.yandex.charset;

import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CoderResult;
import java.nio.charset.CodingErrorAction;

public class StreamEncoder extends Writer {
    public static final int DEFAULT_BUFFER_SIZE = 4096;

    private final OutputStream out;
    private final CharsetEncoder encoder;
    private final byte[] buf;
    private final ByteBuffer bb;
    private final char[] cbuf;
    private final CharBuffer cb;
    private boolean closed = false;

    public StreamEncoder(final OutputStream out, final Charset charset) {
        this(
            out,
            charset.newEncoder()
                .onMalformedInput(CodingErrorAction.REPORT)
                .onUnmappableCharacter(CodingErrorAction.REPORT));
    }

    public StreamEncoder(
        final OutputStream out,
        final CharsetEncoder encoder)
    {
        this(out, encoder, DEFAULT_BUFFER_SIZE);
    }

    public StreamEncoder(
        final OutputStream out,
        final CharsetEncoder encoder,
        final int bufferSize)
    {
        this.out = out;
        this.encoder = encoder;
        buf = new byte[bufferSize];
        bb = ByteBuffer.wrap(buf);
        cbuf = new char[bufferSize];
        cb = CharBuffer.wrap(new char[bufferSize]);
    }

    private void ensureOpen() throws IOException {
        if (closed) {
            throw new IOException("Stream closed");
        }
    }

    @Override
    public void close() throws IOException {
        if (!closed) {
            closed = true;
            cb.flip();
            while (true) {
                CoderResult result = encoder.encode(cb, bb, true);
                if (result.isUnderflow()) {
                    writeBytes();
                    break;
                } else if (result.isOverflow()) {
                    writeBytes();
                } else {
                    result.throwException();
                }
            }
            while (true) {
                CoderResult result = encoder.flush(bb);
                if (result.isUnderflow()) {
                    writeBytes();
                    break;
                } else if (result.isOverflow()) {
                    writeBytes();
                } else {
                    result.throwException();
                }
            }
            out.close();
        }
    }

    @Override
    public void flush() throws IOException {
        ensureOpen();
        cb.flip();
        doWrite(cb);
        cb.compact();
        out.flush();
    }

    @Override
    public void write(final char[] cbuf, int off, int len)
        throws IOException
    {
        ensureOpen();
        // Intentionally use >= compare instead of > so we always have room for
        // leftover char
        if (len >= this.cbuf.length) {
            cb.flip();
            doWrite(cb);
            if (cb.remaining() == 1) {
                cb.compact();
                cb.put(cbuf[off]);
                ++off;
                --len;
                cb.flip();
                doWrite(cb);
            }
            cb.clear();
            CharBuffer cb = CharBuffer.wrap(cbuf, off, len);
            doWrite(cb);
            if (cb.remaining() == 1) {
                this.cb.put(cb.get());
            }
        } else {
            if (len > cb.remaining()) {
                cb.flip();
                doWrite(cb);
                cb.compact();
            }
            cb.put(cbuf, off, len);
        }
    }

    @Override
    public void write(final int c) throws IOException {
        ensureOpen();
        if (cb.remaining() == 0) {
            cb.flip();
            doWrite(cb);
            cb.compact();
        }
        cb.put((char) c);
    }

    private void doWrite(final CharBuffer cb) throws IOException {
        while (cb.hasRemaining()) {
            CoderResult result = encoder.encode(cb, bb, false);
            if (result.isUnderflow()) {
                writeBytes();
                break;
            } else if (result.isOverflow()) {
                writeBytes();
            } else {
                result.throwException();
            }
        }
    }

    private void writeBytes() throws IOException {
        bb.flip();
        int remaining = bb.remaining();
        if (remaining > 0) {
            out.write(buf, bb.position(), remaining);
        }
        bb.clear();
    }
}

