package ru.yandex.charset;

import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;

import ru.yandex.function.ByteArrayProcessor;

// Fast and lightweight implementation for byte[] to String conversion
// intended to replace java.lang.StringCoding
public class StringDecoder
    implements ByteArrayProcessor<String, RuntimeException>
{
    private static final String EMPTY_STRING = "";
    private static final int DEFAULT_MAX_CAPACITY = 1024;
    private static final int MAX_CAPACITY_20KB = 20480;

    public static final ThreadLocal<StringDecoder> UTF_8 =
        ThreadLocal.withInitial(
            () -> new StringDecoder(
                DEFAULT_MAX_CAPACITY,
                StandardCharsets.UTF_8));

    public static final ThreadLocal<StringDecoder> UTF_8_20KB =
        ThreadLocal.withInitial(
            () -> new StringDecoder(
                MAX_CAPACITY_20KB,
                StandardCharsets.UTF_8));

    private final int maxCapacity;
    private final Decoder decoder;

    public StringDecoder(final int maxCapacity, final Charset charset) {
        this.maxCapacity = maxCapacity;
        decoder = new Decoder(
            charset.newDecoder()
                .onMalformedInput(CodingErrorAction.REPLACE)
                .onUnmappableCharacter(CodingErrorAction.REPLACE));
    }

    @Override
    public String process(final byte[] buf, final int off, final int len) {
        if (len == 0) {
            return EMPTY_STRING;
        } else {
            try {
                decoder.process(buf, off, len);
            } catch (CharacterCodingException e) {
                // This can't be because substitution always enabled
                throw new RuntimeException(e);
            }
            String result = decoder.toString();
            decoder.ensureMaxBufCapacity(maxCapacity);
            return result;
        }
    }
}

