package ru.yandex.collection;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.PrimitiveIterator;

public class BinaryTrieMap<V> {
    private static final int NODE_SIZE = 3;
    private static final int INITIAL_NODES_CAPACITY = 512;

    private List<V> values = new ArrayList<>();
    private int[] data = new int[NODE_SIZE * INITIAL_NODES_CAPACITY];
    private int nodes = NODE_SIZE;

    public BinaryTrieMap() {
        values.add(null);
    }

    public V put(final PrimitiveIterator.OfInt bits, final V value) {
        return put(bits, 0, value);
    }

    private V put(
        final PrimitiveIterator.OfInt bits,
        final int pos,
        final V value)
    {
        if (bits.hasNext()) {
            int nodeIndex = pos + bits.nextInt();
            int next = data[nodeIndex];
            if (next == 0) {
                data[nodeIndex] = nodes;
                next = nodes;
                if (nodes >= data.length) {
                    data = Arrays.copyOf(
                        data,
                        ((nodes + (nodes >>> 1)) / NODE_SIZE + 1) * NODE_SIZE);
                }
                nodes += NODE_SIZE;
            }
            return put(bits, next, value);
        } else {
            int dataIndex = pos + 2;
            int oldValueIndex = data[dataIndex];
            V oldValue;
            if (oldValueIndex == 0) {
                oldValue = null;
                data[dataIndex] = values.size();
                values.add(value);
            } else {
                oldValue = values.set(oldValueIndex, value);
            }
            return oldValue;
        }
    }

    public V get(final PrimitiveIterator.OfInt bits) {
        return get(bits, 0);
    }

    // CSOFF: ReturnCount
    private V get(final PrimitiveIterator.OfInt bits, final int pos) {
        if (bits.hasNext()) {
            int next = data[pos + bits.nextInt()];
            if (next == 0) {
                return null;
            } else {
                return get(bits, next);
            }
        } else {
            return values.get(data[pos + 2]);
        }
    }
    // CSON: ReturnCount

    public V getShallowest(final PrimitiveIterator.OfInt bits) {
        return getShallowest(bits, 0);
    }

    // CSOFF: ReturnCount
    private V getShallowest(
        final PrimitiveIterator.OfInt bits,
        final int pos)
    {
        int valueIndex = data[pos + 2];
        if (valueIndex == 0) {
            if (bits.hasNext()) {
                int next = data[pos + bits.nextInt()];
                if (next == 0) {
                    return null;
                } else {
                    return getShallowest(bits, next);
                }
            } else {
                return null;
            }
        } else {
            return values.get(valueIndex);
        }
    }
    // CSON: ReturnCount

    public V getDeepest(final PrimitiveIterator.OfInt bits) {
        return getDeepest(bits, 0, null);
    }

    // CSOFF: ReturnCount
    public V getDeepest(
        final PrimitiveIterator.OfInt bits,
        final int pos,
        final V value)
    {
        V newValue;
        int valueIndex = data[pos + 2];
        if (valueIndex == 0) {
            newValue = value;
        } else {
            newValue = values.get(valueIndex);
        }
        if (bits.hasNext()) {
            int next = data[pos + bits.nextInt()];
            if (next == 0) {
                return newValue;
            } else {
                return getDeepest(bits, next, newValue);
            }
        } else {
            return newValue;
        }
    }
    // CSON: ReturnCount

    @Override
    public String toString() {
        return Arrays.toString(Arrays.copyOfRange(data, 0, nodes)) + values;
    }
}

