package ru.yandex.collection;

import java.util.AbstractList;
import java.util.Arrays;
import java.util.NoSuchElementException;
import java.util.PrimitiveIterator;

// Stores contiguous set of ints just like regular List<Integer>, splitting
// data in chunks of 2KB each (2⁹ ints in each chunk)
public final class ChunkedIntList extends AbstractList<Integer> {
    public static final int CHUNK_SIZE_BITS = 9;
    public static final int CHUNK_SIZE = 1 << CHUNK_SIZE_BITS;
    private static final int MASK = CHUNK_SIZE - 1;
    private static final int DEFAULT_INITIAL_CAPACITY = 16;
    private static final int[] EMPTY_CHUNK = new int[0];
    private static final int[][] EMPTY_CHUNKS = new int[0][];

    private int size = 0;
    private int chunksCount = 0;
    private int[] lastChunk = EMPTY_CHUNK;
    private int initialChunkCapacity;
    private int[][] chunks;
    private boolean shrinked = false;

    public ChunkedIntList() {
        this(0);
    }

    public ChunkedIntList(final int initialCapacity) {
        this(initialCapacity, false);
    }

    public ChunkedIntList(final int initialCapacity, final boolean zeroFill) {
        if (initialCapacity <= 0) {
            initialChunkCapacity = DEFAULT_INITIAL_CAPACITY;
            chunks = EMPTY_CHUNKS;
        } else {
            initialChunkCapacity = Math.min(initialCapacity, CHUNK_SIZE);
            int chunksRequired = (initialCapacity + MASK) >> CHUNK_SIZE_BITS;
            chunks = new int[chunksRequired][];
            if (zeroFill) {
                for (int i = 0; i < chunksRequired - 1; i++) {
                    chunks[i] = new int[CHUNK_SIZE];
                }
                chunksCount = chunksRequired - 1;
                lastChunk = new int[initialChunkCapacity];
                chunks[chunksCount++] = lastChunk;
                size = initialCapacity;
            }
        }
    }

    private void rangeCheck(final int index) {
        if (index >= size) {
            throw new IndexOutOfBoundsException(
                "size: " + size + ", index: " + index);
        }
    }

    public void resize(final int newSize) {
        if (newSize <= size) {
            size = Math.max(0, newSize);
            chunksCount = (size + MASK) >> CHUNK_SIZE_BITS;
            if (chunksCount == 0) {
                lastChunk = EMPTY_CHUNK;
                shrinked = false;
            } else {
                lastChunk = chunks[chunksCount - 1];
                shrinked = (size & MASK) != 0;
            }
        } else {
            int newChunksCount = (newSize + MASK) >> CHUNK_SIZE_BITS;
            int newLastChunkSize = ((newSize - 1) & MASK) + 1;
            if (chunksCount < newChunksCount) {
                if (chunksCount == 0) {
                    chunks = new int[newChunksCount][];
                } else {
                    if (shrinked) {
                        Arrays.fill(
                            lastChunk,
                            size & MASK,
                            lastChunk.length,
                            0);
                        shrinked = false;
                    }
                    chunks[chunksCount - 1] =
                        Arrays.copyOf(lastChunk, CHUNK_SIZE);
                    if (newChunksCount > chunks.length) {
                        chunks = Arrays.copyOf(
                            chunks,
                            Math.max(newChunksCount, chunks.length << 1));
                    }
                }
                while (chunksCount < newChunksCount - 1) {
                    chunks[chunksCount++] = new int[CHUNK_SIZE];
                }
                lastChunk = new int[newLastChunkSize];
                chunks[chunksCount++] = lastChunk;
            } else { // chunksCount == newChunksCount && chunksCount > 0
                if (shrinked) {
                    Arrays.fill(
                        lastChunk,
                        size & MASK,
                        lastChunk.length,
                        0);
                    shrinked = false;
                }
                lastChunk = Arrays.copyOf(lastChunk, newLastChunkSize);
                chunks[chunksCount - 1] = lastChunk;
            }
            size = newSize;
        }
    }

    @Override
    public void clear() {
        size = 0;
        chunksCount = 0;
        lastChunk = EMPTY_CHUNK;
        chunks = EMPTY_CHUNKS;
        shrinked = false;
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public boolean add(final Integer element) {
        addInt(element);
        return true;
    }

    public void addInt(final int element) {
        int pos = size & MASK;
        if (pos == 0) {
            // it is time to add new chunk
            lastChunk = new int[initialChunkCapacity];
            // next chunk initial capacity should be bigger
            initialChunkCapacity =
                Math.min(initialChunkCapacity << 1, CHUNK_SIZE);
            if (chunksCount == chunks.length) {
                if (chunksCount == 0) {
                    chunks = new int[DEFAULT_INITIAL_CAPACITY][];
                } else {
                    chunks = Arrays.copyOf(chunks, chunksCount << 1);
                }
            }
            chunks[chunksCount++] = lastChunk;
        } else if (pos == lastChunk.length) {
            // pos < CHUNK_SIZE
            lastChunk =
                Arrays.copyOf(lastChunk, Math.min(pos << 1, CHUNK_SIZE));
            chunks[chunksCount - 1] = lastChunk;
        }
        lastChunk[pos] = element;
        ++size;
    }

    @Override
    public Integer get(final int index) {
        return getInt(index);
    }

    public int getInt(final int index) {
        rangeCheck(index);
        return chunks[index >> CHUNK_SIZE_BITS][index & MASK];
    }

    @Override
    public Integer set(final int index, final Integer element) {
        return setInt(index, element);
    }

    public int setInt(final int index, final int element) {
        rangeCheck(index);
        int[] chunk = chunks[index >> CHUNK_SIZE_BITS];
        int chunkIndex = index & MASK;
        int result = chunk[chunkIndex];
        chunk[chunkIndex] = element;
        return result;
    }

    public int inc(final int index) {
        rangeCheck(index);
        return chunks[index >> CHUNK_SIZE_BITS][index & MASK]++;
    }

    public int pinc(final int index) {
        rangeCheck(index);
        return ++chunks[index >> CHUNK_SIZE_BITS][index & MASK];
    }

    @Override
    public PrimitiveIterator.OfInt iterator() {
        return new Itr();
    }

    private class Itr implements PrimitiveIterator.OfInt {
        private int pos = 0;

        @Override
        public boolean hasNext() {
            return pos < size;
        }

        @Override
        public int nextInt() {
            if (pos >= size) {
                throw new NoSuchElementException();
            }
            int result = chunks[pos >> CHUNK_SIZE_BITS][pos & MASK];
            ++pos;
            return result;
        }
    }
}

