package ru.yandex.collection;

import java.util.AbstractList;
import java.util.Arrays;
import java.util.NoSuchElementException;

// Stores contiguous set of longs just like regular List<Long>, splitting
// data in chunks of 4KB each (2⁹ longs in each chunk) by default
public class ChunkedLongList extends AbstractList<Long> {
    private static final int DEFAULT_CHUNK_SIZE_BITS = 9;
    private static final int DEFAULT_INITIAL_CAPACITY = 16;
    private static final long[] EMPTY_CHUNK = new long[0];
    private static final long[][] EMPTY_CHUNKS = new long[0][];

    private final int chunkSizeBits;
    private final int chunkSize;
    private final int mask;
    private int size = 0;
    private int chunksCount = 0;
    private long[] lastChunk = EMPTY_CHUNK;
    private int initialChunkCapacity;
    private long[][] chunks;
    private boolean shrinked = false;

    public ChunkedLongList() {
        this(0);
    }

    public ChunkedLongList(final int initialCapacity) {
        this(initialCapacity, DEFAULT_CHUNK_SIZE_BITS);
    }

    public ChunkedLongList(
        final int initialCapacity,
        final int chunkSizeBits)
    {
        this.chunkSizeBits = chunkSizeBits;
        chunkSize = 1 << chunkSizeBits;
        mask = chunkSize - 1;
        if (initialCapacity <= 0) {
            initialChunkCapacity = DEFAULT_INITIAL_CAPACITY;
            chunks = EMPTY_CHUNKS;
        } else {
            initialChunkCapacity = Math.min(initialCapacity, chunkSize);
            int chunksRequired = (initialCapacity + mask) >> chunkSizeBits;
            chunks = new long[chunksRequired][];
        }
    }

    private void rangeCheck(final int index) {
        if (index >= size) {
            throw new IndexOutOfBoundsException(
                "size: " + size + ", index: " + index);
        }
    }

    public void resize(final int newSize) {
        if (newSize <= size) {
            size = Math.max(0, newSize);
            chunksCount = (size + mask) >> chunkSizeBits;
            if (chunksCount == 0) {
                lastChunk = EMPTY_CHUNK;
                shrinked = false;
            } else {
                lastChunk = chunks[chunksCount - 1];
                shrinked = (size & mask) != 0;
            }
        } else {
            int newChunksCount = (newSize + mask) >> chunkSizeBits;
            int newLastChunkSize = ((newSize - 1) & mask) + 1;
            if (chunksCount < newChunksCount) {
                if (chunksCount == 0) {
                    chunks = new long[newChunksCount][];
                } else {
                    if (shrinked) {
                        Arrays.fill(
                            lastChunk,
                            size & mask,
                            lastChunk.length,
                            0);
                        shrinked = false;
                    }
                    chunks[chunksCount - 1] =
                        Arrays.copyOf(lastChunk, chunkSize);
                    if (newChunksCount > chunks.length) {
                        chunks = Arrays.copyOf(
                            chunks,
                            Math.max(newChunksCount, chunks.length << 1));
                    }
                }
                while (chunksCount < newChunksCount - 1) {
                    chunks[chunksCount++] = new long[chunkSize];
                }
                lastChunk = new long[newLastChunkSize];
                chunks[chunksCount++] = lastChunk;
            } else { // chunksCount == newChunksCount && chunksCount > 0
                if (shrinked) {
                    Arrays.fill(
                        lastChunk,
                        size & mask,
                        lastChunk.length,
                        0);
                    shrinked = false;
                }
                lastChunk = Arrays.copyOf(lastChunk, newLastChunkSize);
                chunks[chunksCount - 1] = lastChunk;
            }
            size = newSize;
        }
    }

    @Override
    public void clear() {
        size = 0;
        chunksCount = 0;
        lastChunk = EMPTY_CHUNK;
        chunks = EMPTY_CHUNKS;
        shrinked = false;
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public boolean add(final Long element) {
        addLong(element);
        return true;
    }

    public void addLong(final long element) {
        int pos = size & mask;
        if (pos == 0) {
            // it is time to add new chunk
            lastChunk = new long[initialChunkCapacity];
            // next chunk initial capacity should be bigger
            initialChunkCapacity =
                Math.min(initialChunkCapacity << 1, chunkSize);
            if (chunksCount == chunks.length) {
                if (chunksCount == 0) {
                    chunks = new long[DEFAULT_INITIAL_CAPACITY][];
                } else {
                    chunks = Arrays.copyOf(chunks, chunksCount << 1);
                }
            }
            chunks[chunksCount++] = lastChunk;
        } else if (pos == lastChunk.length) {
            // pos < chunkSize
            lastChunk =
                Arrays.copyOf(lastChunk, Math.min(pos << 1, chunkSize));
            chunks[chunksCount - 1] = lastChunk;
        }
        lastChunk[pos] = element;
        ++size;
    }

    @Override
    public Long get(final int index) {
        return getLong(index);
    }

    public long getLong(final int index) {
        rangeCheck(index);
        return chunks[index >> chunkSizeBits][index & mask];
    }

    @Override
    public Long set(final int index, final Long element) {
        return setLong(index, element);
    }

    public long setLong(final int index, final long element) {
        rangeCheck(index);
        long[] chunk = chunks[index >> chunkSizeBits];
        int chunkIndex = index & mask;
        long result = chunk[chunkIndex];
        chunk[chunkIndex] = element;
        return result;
    }

    public long inc(final int index) {
        rangeCheck(index);
        return chunks[index >> chunkSizeBits][index & mask]++;
    }

    public long pinc(final int index) {
        rangeCheck(index);
        return ++chunks[index >> chunkSizeBits][index & mask];
    }

    @Override
    public StepBackPrimitiveIterator.OfLong iterator() {
        return new Itr();
    }

    private class Itr implements StepBackPrimitiveIterator.OfLong {
        private int pos = 0;

        @Override
        public boolean hasNext() {
            return pos < size;
        }

        @Override
        public long nextLong() {
            if (pos >= size) {
                throw new NoSuchElementException();
            }
            long result = chunks[pos >> chunkSizeBits][pos & mask];
            ++pos;
            return result;
        }

        @Override
        public void stepBack() {
            if (pos == 0) {
                throw new IllegalStateException("Nowhere to step back");
            }
            --pos;
        }
    }
}

