package ru.yandex.collection;

import java.util.Comparator;
import java.util.Iterator;
import java.util.Objects;

public class CollectionsComparator<T> implements Comparator<Iterable<T>> {
    private static final long serialVersionUID = 0L;
    private static final CollectionsComparator<Comparable<Object>> INSTANCE =
        new CollectionsComparator<>(Comparator.naturalOrder());

    private final Comparator<? super T> comparator;

    public CollectionsComparator(final Comparator<? super T> comparator) {
        this.comparator = comparator;
    }

    public static boolean startsWith(
        final Iterable<?> lhs,
        final Iterable<?> rhs)
    {
        return startsWith(lhs.iterator(), rhs.iterator());
    }

    public static boolean startsWith(
        final Iterator<?> lhs,
        final Iterator<?> rhs)
    {
        while (lhs.hasNext() && rhs.hasNext()) {
            if (!Objects.equals(lhs.next(), rhs.next())) {
                return false;
            }
        }
        return Boolean.compare(lhs.hasNext(), rhs.hasNext()) >= 0;
    }

    @SuppressWarnings("unchecked")
    public static <T extends Comparable<? super T>> CollectionsComparator<T>
        naturalOrder()
    {
        return (CollectionsComparator<T>) INSTANCE;
    }

    @Override
    public int compare(final Iterable<T> lhs, final Iterable<T> rhs) {
        return compare(lhs.iterator(), rhs.iterator());
    }

    public int compare(final Iterator<T> lhs, final Iterator<T> rhs) {
        while (lhs.hasNext() && rhs.hasNext()) {
            int cmp = comparator.compare(lhs.next(), rhs.next());
            if (cmp != 0) {
                return cmp;
            }
        }
        return Boolean.compare(lhs.hasNext(), rhs.hasNext());
    }

    @Override
    public int hashCode() {
        return comparator.hashCode();
    }

    @Override
    public boolean equals(final Object o) {
        return o instanceof CollectionsComparator
            && comparator.equals(((CollectionsComparator) o).comparator);
    }
}

