package ru.yandex.collection;

import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Set;

public class IdentityHashSet<E> implements Set<E> {
    private final IdentityHashMap<E, E> map;

    public IdentityHashSet() {
        map = new IdentityHashMap<>();
    }

    public IdentityHashSet(final int expectedMaxSize) {
        map = new IdentityHashMap<>(expectedMaxSize);
    }

    public IdentityHashSet(final Collection<? extends E> c) {
        map = new IdentityHashMap<>(c.size() << 1);
        for (E e: c) {
            map.put(e, e);
        }
    }

    @Override
    public boolean add(final E e) {
        int size = map.size();
        map.put(e, e);
        return size != map.size();
    }

    @Override
    public boolean addAll(final Collection<? extends E> c) {
        int size = map.size();
        for (E e: c) {
            map.put(e, e);
        }
        return size != map.size();
    }

    @Override
    public void clear() {
        map.clear();
    }

    @Override
    public boolean contains(final Object o) {
        return map.containsKey(o);
    }

    @Override
    public boolean containsAll(final Collection<?> c) {
        for (Object o: c) {
            if (!map.containsKey(o)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean equals(final Object o) {
        boolean result = false;
        if (o instanceof Set) {
            Set<?> other = (Set<?>) o;
            if (map.size() == other.size()) {
                try {
                    result = containsAll(other);
                } catch (ClassCastException | NullPointerException e) {
                    // ignore
                }
            }
        }
        return result;
    }

    @Override
    public int hashCode() {
        // IdentityHashMap.keySet().hashCode() implementation differs from
        // AbstractSet.hashCode() implementation, so we have to reimplement it
        int h = 0;
        for (E e: map.keySet()) {
            if (e != null) {
                h += e.hashCode();
            }
        }
        return h;
    }

    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    @Override
    public Iterator<E> iterator() {
        return map.keySet().iterator();
    }

    @Override
    public boolean remove(final Object o) {
        int size = map.size();
        map.remove(o);
        return size != map.size();
    }

    @Override
    public boolean removeAll(final Collection<?> c) {
        int size = map.size();
        for (Object o: c) {
            map.remove(o);
        }
        return size != map.size();
    }

    @Override
    public boolean retainAll(final Collection<?> c) {
        return map.keySet().retainAll(c);
    }

    @Override
    public int size() {
        return map.size();
    }

    @Override
    public Object[] toArray() {
        return map.keySet().toArray();
    }

    @Override
    public <T> T[] toArray(final T[] a) {
        return map.keySet().toArray(a);
    }

    @Override
    public String toString() {
        return map.keySet().toString();
    }
}

