package ru.yandex.collection;

import java.util.BitSet;

public class IntInterval implements Comparable<IntInterval>, IntSet {
    private static final int HASH_ROTATE = 13;

    private final int min;
    private final int max;

    public IntInterval(final int min, final int max) {
        this.min = min;
        this.max = max;
        if (min > max) {
            throw new IllegalArgumentException(
                "Interval start must be not greater than interval end: "
                + this);
        }
    }

    @Override
    public int min() {
        return min;
    }

    @Override
    public int max() {
        return max;
    }

    public IntInterval mergeIfOverlaps(final IntInterval other) {
        IntInterval result = null;
        if (min <= other.min) {
            if (max >= other.max) {
                result = this;
            } else if (max + 1 >= other.min) {
                result = new IntInterval(min, other.max);
            } else {
                result = null;
            }
        } else if (max <= other.max) {
            result = other;
        } else if (min - 1 <= other.max) {
            result = new IntInterval(other.min, max);
        } else {
            result = null;
        }
        return result;
    }

    @Override
    public boolean contains(final int value) {
        return value >= min && value <= max;
    }

    @Override
    public int compareTo(final IntInterval other) {
        int cmp = Integer.compare(min, other.min);
        if (cmp == 0) {
            cmp = Integer.compare(max, other.max);
        }
        return cmp;
    }

    @Override
    public IntIntervalIterator iterator() {
        return new IntIntervalIterator(min, max);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(min);
        sb.append('-');
        sb.append(max);
        return new String(sb);
    }

    @Override
    public int hashCode() {
        return Integer.rotateLeft(min, HASH_ROTATE) ^ max;
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof IntInterval) {
            IntInterval other = (IntInterval) o;
            return min == other.min && max == other.max;
        }
        return false;
    }

    @Override
    public BitSet toBitSet() {
        BitSet bitSet = new BitSet(max + 1);
        toBitSet(bitSet);
        return bitSet;
    }

    public void toBitSet(final BitSet bitSet) {
        bitSet.set(min, max + 1);
    }
}

