package ru.yandex.collection;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.PrimitiveIterator;

public class IntIntervalSet implements IntSet {
    private final IntInterval[] intervals;

    protected IntIntervalSet(final IntInterval[] intervals) {
        this.intervals = intervals;
    }

    public static IntSet create(final BitSet bitSet) {
        List<IntInterval> intervals = new ArrayList<>();
        int end = -1;
        while (true) {
            int start = bitSet.nextSetBit(end + 1);
            if (start == -1) {
                break;
            }
            end = bitSet.nextClearBit(start + 1);
            intervals.add(new IntInterval(start, end - 1));
        }
        return create(intervals);
    }

    public static IntSet create(final List<IntInterval> intervals) {
        IntSet intSet;
        if (intervals.isEmpty()) {
            intSet = EmptyIntSet.INSTANCE;
        } else {
            List<IntInterval> result = new ArrayList<>(intervals.size());
            Collections.sort(intervals);
            Iterator<IntInterval> iter = intervals.iterator();
            IntInterval current = iter.next();
            while (iter.hasNext()) {
                IntInterval next = iter.next();
                IntInterval merged = current.mergeIfOverlaps(next);
                if (merged == null) {
                    result.add(current);
                    current = next;
                } else {
                    current = merged;
                }
            }
            if (result.isEmpty()) {
                intSet = current;
            } else {
                result.add(current);
                intSet = new IntIntervalSet(
                    result.toArray(new IntInterval[result.size()]));
            }
        }
        return intSet;
    }

    @Override
    public int min() {
        return intervals[0].min();
    }

    @Override
    public int max() {
        return intervals[intervals.length - 1].max();
    }

    @Override
    public boolean contains(final int value) {
        int from = 0;
        int to = intervals.length - 1;
        while (from <= to) {
            int i = (from + to) >>> 1;
            IntInterval interval = intervals[i];
            if (interval.min() > value) {
                to = i - 1;
            } else if (interval.max() < value) {
                from = i + 1;
            } else {
                return true;
            }
        }
        return false;
    }

    @Override
    public PrimitiveIterator.OfInt iterator() {
        return new Itr(intervals);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(intervals[0].min());
        sb.append('-');
        sb.append(intervals[0].max());
        for (int i = 1; i < intervals.length; ++i) {
            sb.append(',');
            sb.append(intervals[i].min());
            sb.append('-');
            sb.append(intervals[i].max());
        }
        return new String(sb);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(intervals);
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof IntIntervalSet) {
            return Arrays.equals(intervals, ((IntIntervalSet) o).intervals);
        }
        return false;
    }

    @Override
    public BitSet toBitSet() {
        BitSet bitSet = new BitSet(max() + 1);
        for (IntInterval interval: intervals) {
            interval.toBitSet(bitSet);
        }
        return bitSet;
    }

    private static class Itr
        implements PrimitiveIterator.OfInt
    {
        private final IntInterval[] intervals;
        private int pos = 1;
        private PrimitiveIterator.OfInt current;

        // None of intervals are empty and there are at least 1 interval
        // present. Actually, there will be at least two intervals
        Itr(final IntInterval[] intervals) {
            this.intervals = intervals;
            current = intervals[0].iterator();
        }

        @Override
        public boolean hasNext() {
            return current.hasNext();
        }

        @Override
        public int nextInt() {
            if (hasNext()) {
                int next = current.nextInt();
                if (!current.hasNext() && pos < intervals.length) {
                    current = intervals[pos++].iterator();
                }
                return next;
            } else {
                throw new NoSuchElementException();
            }
        }

        @Override
        public Integer next() {
            return nextInt();
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}

