package ru.yandex.collection;

import java.util.AbstractList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.PrimitiveIterator;

import ru.yandex.function.StringBuilderable;

public class IntList
    extends AbstractList<Integer>
    implements StringBuilderable
{
    private static final int DEFAULT_INITIAL_CAPACITY = 16;
    private static final int[] EMPTY_DATA = new int[0];
    // Will be enough for most cases, sacrifice realloc or two for the sake of
    // memory consumption. This includes delimiter between elements
    private static final int EXPECTED_CHARS_PER_ELEMENT = 6;

    private int size = 0;
    private int[] data;

    public IntList() {
        this(0);
    }

    public IntList(final int initialCapacity) {
        if (initialCapacity <= 0) {
            data = EMPTY_DATA;
        } else {
            data = new int[initialCapacity];
        }
    }

    private void rangeCheck(final int index) {
        if (index >= size) {
            throw new IndexOutOfBoundsException(index + " >= " + size);
        }
    }

    // fills new space with zeroes
    public void resize(final int newSize) {
        if (newSize <= size) {
            size = Math.max(newSize, 0);
        } else {
            int len = data.length;
            Arrays.fill(data, size, len, 0);
            if (newSize > len) {
                data = Arrays.copyOf(data, Math.max(len << 1, newSize));
            }
            size = newSize;
        }
    }

    public void shrink() {
        if (size < data.length) {
            data = Arrays.copyOf(data, size);
        }
    }

    public void sort() {
        Arrays.sort(data, 0, size);
    }

    // data expected to be sorted
    public int binarySearch(final int key) {
        if (size > 0) {
            return Arrays.binarySearch(data, 0, size, key);
        } else {
            return -1;
        }
    }

    @Override
    public void clear() {
        size = 0;
        data = EMPTY_DATA;
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public boolean contains(final Object o) {
        return o instanceof Integer && containsInt(((Integer) o).intValue());
    }

    public boolean containsInt(final int value) {
        for (int i = 0; i < size; ++i) {
            if (data[i] == value) {
                return true;
            }
        }
        return false;
    }

    @Override
    public Integer get(final int index) {
        return getInt(index);
    }

    public int getInt(final int index) {
        rangeCheck(index);
        return data[index];
    }

    @Override
    public boolean add(final Integer element) {
        return addInt(element);
    }

    public boolean addInt(final int element) {
        if (size == data.length) {
            if (size <= 0) {
                data = new int[DEFAULT_INITIAL_CAPACITY];
            } else {
                data = Arrays.copyOf(data, size << 1);
            }
        }
        data[size++] = element;
        return true;
    }

    @Override
    public Integer set(final int index, final Integer element) {
        return setInt(index, element);
    }

    public int setInt(final int index, final int element) {
        rangeCheck(index);
        int result = data[index];
        data[index] = element;
        return result;
    }

    @Override
    public Integer remove(final int index) {
        return removeInt(index);
    }

    public int removeInt(final int index) {
        rangeCheck(index);
        int result = data[index];
        int copyLen = --size - index;
        if (copyLen > 0) {
            System.arraycopy(data, index + 1, data, index, copyLen);
        }
        return result;
    }

    public int removeLast() {
        if (size > 0) {
            --size;
            return data[size];
        } else {
            throw new IndexOutOfBoundsException("IntList already empty");
        }
    }

    @Override
    public PrimitiveIterator.OfInt iterator() {
        return new Itr();
    }

    @Override
    public int hashCode() {
        int hash = 1;
        for (int i = 0; i < size; ++i) {
            hash = 31 * hash + data[i];
        }
        return hash;
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof IntList) {
            IntList other = (IntList) o;
            return size == other.size
                && Arrays.equals(data, 0, size, other.data, 0, size);
        } else if (o instanceof List) {
            List<?> other = (List<?>) o;
            if (size == other.size()) {
                Iterator<?> iter = other.iterator();
                int i = 0;
                while (iter.hasNext()) {
                    Object e = iter.next();
                    if (e == null || !e.equals(data[i++])) {
                        return false;
                    }
                }
                return true;
            } else {
                return false;
            }
        }
        return false;
    }

    @Override
    public int expectedStringLength() {
        if (size <= 0) {
            return 2;
        } else {
            return size * EXPECTED_CHARS_PER_ELEMENT + 2;
        }
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append('[');
        if (size > 0) {
            sb.append(data[0]);
            for (int i = 1; i < size; ++i) {
                sb.append(',');
                sb.append(' ');
                sb.append(data[i]);
            }
        }
        sb.append(']');
    }

    @Override
    public String toString() {
        return toStringFast();
    }

    public int[] toIntArray() {
        if (size <= 0) {
            return EMPTY_DATA;
        } else {
            return Arrays.copyOf(data, size);
        }
    }

    private class Itr implements PrimitiveIterator.OfInt {
        private int pos = 0;

        @Override
        public boolean hasNext() {
            return pos < size;
        }

        @Override
        public int nextInt() {
            if (pos >= size) {
                throw new NoSuchElementException();
            }
            return data[pos++];
        }
    }
}

