package ru.yandex.collection;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;

public final class Iterators {
    private static final int DEFAULT_LIST_CAPACITY = 4;

    // 11 prime numbers > 1000, such LAST - FIRST -> min
    private static final List<Integer> PRIMES =
        Collections.unmodifiableList(Arrays.asList(
            51448351,
            51448357,
            51448361,
            51448363,
            51448367,
            51448373,
            51448379,
            51448381,
            51448387,
            51448391,
            51448393));

    private Iterators() {
    }

    public static <T> Iterator<T> randomIterator(
        final List<? extends T> items,
        final int seedValue)
    {
        if (items.isEmpty()) {
            return Collections.emptyIterator();
        }

        final int seed;
        if (seedValue >= 0) {
            seed = seedValue;
        } else {
            seed = ~seedValue;
        }

        final int size = items.size();
        final int shift = PRIMES.get(seed % PRIMES.size()) % size;

        return new Iterator<T>() {
            private int left = size;
            private int next = seed % size;

            @Override
            public boolean hasNext() {
                return left > 0;
            }

            @Override
            public T next() {
                if (!hasNext()) {
                    throw new NoSuchElementException();
                }
                T value = items.get(next);
                --left;
                next += shift;
                if (next >= size) {
                    next -= size;
                }
                return value;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    public static <T> List<T> toList(final Iterator<? extends T> iterator) {
        return toList(iterator, DEFAULT_LIST_CAPACITY);
    }

    public static <T> List<T> toList(
        final Iterator<? extends T> iterator,
        final int capacity)
    {
        List<T> result = new ArrayList<>(capacity);
        while (iterator.hasNext()) {
            result.add(iterator.next());
        }
        return result;
    }

    public static boolean equals(
        final Iterator<?> lhs,
        final Iterator<?> rhs)
    {
        while (lhs.hasNext() && rhs.hasNext()) {
            if (!Objects.equals(lhs.next(), rhs.next())) {
                return false;
            }
        }
        return lhs.hasNext() == rhs.hasNext();
    }

    public static int size(final Iterator<?> iter) {
        int size = 0;
        while (iter.hasNext()) {
            iter.next();
            ++size;
        }
        return size;
    }
}

