package ru.yandex.collection;

import java.util.AbstractList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.PrimitiveIterator;

import ru.yandex.function.StringBuilderable;

public class LongList extends AbstractList<Long> implements StringBuilderable {
    private static final int DEFAULT_INITIAL_CAPACITY = 16;
    private static final long[] EMPTY_DATA = new long[0];
    // Will be enough for most cases, sacrifice realloc or two for the sake of
    // memory consumption. This includes delimiter between elements
    private static final int EXPECTED_CHARS_PER_ELEMENT = 6;

    private int size = 0;
    private long[] data;

    public LongList() {
        this(0);
    }

    public LongList(final int initialCapacity) {
        if (initialCapacity <= 0) {
            data = EMPTY_DATA;
        } else {
            data = new long[initialCapacity];
        }
    }

    public static LongList of(final long value) {
        LongList list = new LongList(1);
        list.size = 1;
        list.data[0] = value;
        return list;
    }

    private void rangeCheck(final int index) {
        if (index >= size) {
            throw new IndexOutOfBoundsException(index + " >= " + size);
        }
    }

    // fills new space with zeroes
    public void resize(final int newSize) {
        if (newSize <= size) {
            size = Math.max(newSize, 0);
        } else {
            int len = data.length;
            Arrays.fill(data, size, len, 0);
            if (newSize > len) {
                data = Arrays.copyOf(data, Math.max(len << 1, newSize));
            }
            size = newSize;
        }
    }

    public void shrink() {
        if (size < data.length) {
            data = Arrays.copyOf(data, size);
        }
    }

    public void sort() {
        Arrays.sort(data, 0, size);
    }

    // data expected to be sorted
    public int binarySearch(final long key) {
        if (size > 0) {
            return Arrays.binarySearch(data, 0, size, key);
        } else {
            return -1;
        }
    }

    @Override
    public void clear() {
        size = 0;
        data = EMPTY_DATA;
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public boolean contains(final Object o) {
        return o instanceof Long && containsLong(((Long) o).longValue());
    }

    public boolean containsLong(final long value) {
        for (int i = 0; i < size; ++i) {
            if (data[i] == value) {
                return true;
            }
        }
        return false;
    }

    @Override
    public Long get(final int index) {
        return getLong(index);
    }

    public long getLong(final int index) {
        rangeCheck(index);
        return data[index];
    }

    @Override
    public boolean add(final Long element) {
        return addLong(element);
    }

    public boolean addLong(final long element) {
        if (size == data.length) {
            if (size <= 0) {
                data = new long[DEFAULT_INITIAL_CAPACITY];
            } else {
                data = Arrays.copyOf(data, size << 1);
            }
        }
        data[size++] = element;
        return true;
    }

    public boolean addLongs(final long[] source, int offset, int count) {
        if (size + count > data.length) {
            if (size <= 0) {
                data = new long[Math.max(DEFAULT_INITIAL_CAPACITY, count)];
            } else {
                data = Arrays.copyOf(data, Math.max(size << 1, size + count));
            }
        }
        System.arraycopy(source, offset, data, size, count);
        size += count;
        return true;
    }

    @Override
    public Long set(final int index, final Long element) {
        return setLong(index, element);
    }

    public long setLong(final int index, final long element) {
        rangeCheck(index);
        long result = data[index];
        data[index] = element;
        return result;
    }

    @Override
    public Long remove(final int index) {
        return removeLong(index);
    }

    public long removeLong(final int index) {
        rangeCheck(index);
        long result = data[index];
        int copyLen = --size - index;
        if (copyLen > 0) {
            System.arraycopy(data, index + 1, data, index, copyLen);
        }
        return result;
    }

    public long removeLast() {
        if (size > 0) {
            --size;
            return data[size];
        } else {
            throw new IndexOutOfBoundsException("LongList already empty");
        }
    }

    @Override
    public PrimitiveIterator.OfLong iterator() {
        return new Itr();
    }

    @Override
    public int hashCode() {
        int hash = 1;
        for (int i = 0; i < size; ++i) {
            long value = data[i];
            hash = 31 * hash + (int) (value ^ (value >>> 32));
        }
        return hash;
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof LongList) {
            LongList other = (LongList) o;
            return size == other.size
                && Arrays.equals(data, 0, size, other.data, 0, size);
        } else if (o instanceof List) {
            List<?> other = (List<?>) o;
            if (size == other.size()) {
                Iterator<?> iter = other.iterator();
                int i = 0;
                while (iter.hasNext()) {
                    Object e = iter.next();
                    if (e == null || !e.equals(data[i++])) {
                        return false;
                    }
                }
                return true;
            } else {
                return false;
            }
        }
        return false;
    }

    @Override
    public int expectedStringLength() {
        if (size <= 0) {
            return 2;
        } else {
            return size * EXPECTED_CHARS_PER_ELEMENT + 2;
        }
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append('[');
        if (size > 0) {
            sb.append(data[0]);
            for (int i = 1; i < size; ++i) {
                sb.append(',');
                sb.append(' ');
                sb.append(data[i]);
            }
        }
        sb.append(']');
    }

    @Override
    public String toString() {
        return toStringFast();
    }

    public long[] toLongArray() {
        if (size <= 0) {
            return EMPTY_DATA;
        } else {
            return Arrays.copyOf(data, size);
        }
    }

    private class Itr implements PrimitiveIterator.OfLong {
        private int pos = 0;

        @Override
        public boolean hasNext() {
            return pos < size;
        }

        @Override
        public long nextLong() {
            if (pos >= size) {
                throw new NoSuchElementException();
            }
            return data[pos++];
        }
    }
}

