#include "compress.h"

#include <contrib/libs/brotli/include/brotli/encode.h>
#include <contrib/libs/brotli/include/brotli/decode.h>

#include <alloca.h>
#include <errno.h>

#include <jni.h>

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_brotliCompressByteArray
(JNIEnv* env,
jclass class,
jbyteArray input,
jint inputlen,
jbyteArray output,
jint outputlen,
jint quality)
{
    (void) class;
    jint ret;
    jbyte *inputArray, *outputArray = NULL;
    size_t encodedLength = outputlen;
    BROTLI_BOOL ok = 0;
    inputArray = (*env)->GetPrimitiveArrayCritical(env, input, 0);
    if (!inputArray) {
        ret = -100500;
        goto exit;
    }
    outputArray = (*env)->GetPrimitiveArrayCritical(env, output, 0);
    if (!outputArray) {
        ret = -100501;
        goto exit;
    }
    ok = BrotliEncoderCompress(
        quality,
        BROTLI_DEFAULT_WINDOW,
        BROTLI_MODE_GENERIC,
        inputlen,
        (unsigned char*) inputArray,
        &encodedLength,
        (unsigned char*) outputArray);
    if (ok == BROTLI_TRUE) {
        ret = encodedLength;
    } else {
        ret = -1;
    }

exit:
    if (inputArray) {
        (*env)->ReleasePrimitiveArrayCritical(
            env,
            input,
            inputArray,
            JNI_ABORT);
    }
    if (outputArray) {
        (*env)->ReleasePrimitiveArrayCritical(
            env,
            output,
            outputArray,
            ret > 0 ? 0 : JNI_ABORT);
    }
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_brotliCompressRawArray
(JNIEnv* env,
jclass class,
jlong input,
jint inputlen,
jlong output,
jint outputlen,
jint quality)
{
    (void) class;
    (void) env;
    size_t encodedLength = outputlen;
    BROTLI_BOOL ok = BrotliEncoderCompress(
        quality,
        BROTLI_DEFAULT_WINDOW,
        BROTLI_MODE_GENERIC,
        inputlen,
        (unsigned char*) input,
        &encodedLength,
        (unsigned char*) output);
    jint ret;
    if (ok == BROTLI_TRUE) {
        ret = encodedLength;
    } else {
        ret = -1;
    }
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL JavaCritical_ru_yandex_compress_NativeCompress_brotliCompressRawArray
(jlong input,
jint inputlen,
jlong output,
jint outputlen,
jint quality)
{
    size_t encodedLength = outputlen;
    jint ret;
    BROTLI_BOOL ok = BrotliEncoderCompress(
        quality,
        BROTLI_DEFAULT_WINDOW,
        BROTLI_MODE_GENERIC,
        inputlen,
        (unsigned char*) input,
        &encodedLength,
        (unsigned char*) output);
    if (ok == BROTLI_TRUE) {
        ret = encodedLength;
    } else {
        ret = -1;
    }
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_brotliDecompressByteArray
(JNIEnv* env,
jclass class,
jbyteArray input,
jint inputlen,
jlong output,
jint outputlen)
{
    (void) class;
    jbyte *inputArray = (*env)->GetPrimitiveArrayCritical(env, input, 0);
    if (!inputArray) {
        return -100500;
    }
    size_t decodedLength = outputlen;
    BrotliDecoderResult result = BrotliDecoderDecompress(
        inputlen,
        (unsigned char*) inputArray,
        &decodedLength,
        (unsigned char*) output);
    jint ret;
    if (result == BROTLI_DECODER_RESULT_SUCCESS) {
        ret = decodedLength;
    } else {
        ret = -1;
    }
    (*env)->ReleasePrimitiveArrayCritical(
        env,
        input,
        inputArray,
        JNI_ABORT);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_brotliDecompressBytes
(JNIEnv* env,
jclass class,
jlong input,
jint inputlen,
jlong output,
jint outputlen)
{
    (void) class;
    (void) env;
    size_t decodedLength = outputlen;
    BrotliDecoderResult result = BrotliDecoderDecompress(
        inputlen,
        (unsigned char*) input,
        &decodedLength,
        (unsigned char*) output);
    jint ret;
    if (result == BROTLI_DECODER_RESULT_SUCCESS) {
        ret = decodedLength;
    } else {
        ret = -1;
    }
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL JavaCritical_ru_yandex_compress_NativeCompress_brotliDecompressBytes
(jlong input,
jint inputlen,
jlong output,
jint outputlen)
{
    size_t decodedLength = outputlen;
    BrotliDecoderResult result = BrotliDecoderDecompress(
        inputlen,
        (unsigned char*) input,
        &decodedLength,
        (unsigned char*) output);
    jint ret;
    if (result == BROTLI_DECODER_RESULT_SUCCESS) {
        ret = decodedLength;
    } else {
        ret = -1;
    }
    return ret;
}

static jint brotliReadAndDecompressByteArray
(jint fd,
jlong fPos,
jint deflatedlen,
jlong output,
jint outputlen)
{
    void *readBuffer;

    readBuffer = alloca(deflatedlen);
    int read = PreadFull(fd, readBuffer, deflatedlen, fPos);
    if (read == 0 || read < deflatedlen) {
        return EOF_ERROR;
    }
    if (read == -1) {
        return -errno;
    }
    size_t decodedLength = outputlen;
    BrotliDecoderResult result = BrotliDecoderDecompress(
        deflatedlen,
        (unsigned char*) readBuffer,
        &decodedLength,
        (unsigned char*) output);
    jint ret;
    if (result == BROTLI_DECODER_RESULT_SUCCESS) {
        ret = decodedLength;
    } else {
        ret = -1;
    }
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_brotliReadAndDecompressByteArray
(JNIEnv* env,
jclass class,
jint fd,
jlong fPos,
jint deflatedlen,
jlong output,
jint outputlen)
{
    (void) env;
    (void) class;
    return brotliReadAndDecompressByteArray(fd, fPos, deflatedlen, output, outputlen);
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL JavaCritical_ru_yandex_compress_NativeCompress_brotliReadAndDecompressByteArray
(jint fd,
jlong fPos,
jint deflatedlen,
jlong output,
jint outputlen)
{
    return brotliReadAndDecompressByteArray(fd, fPos, deflatedlen, output, outputlen);
}

