#include "compress.h"

#include <contrib/libs/lzmasdk/LzmaLib.h>

#include <alloca.h>
#include <errno.h>
#include <unistd.h>

#include <jni.h>

//LZMA
int LZMA_compress(
    unsigned char* outputArray,
    int *outputlen,
    unsigned char* inputArray,
    int inputlen,
    int level)
{
    unsigned char *prop = outputArray;
    unsigned char *out = prop + LZMA_PROPS_SIZE;
    size_t propSize = LZMA_PROPS_SIZE;
    size_t outSize = *outputlen;
    size_t inSize = inputlen;
    outSize -= LZMA_PROPS_SIZE;
    int ret = LzmaCompress(
        out,
        &outSize,
        inputArray,
        inSize,
        prop,
        &propSize,
        level,
        0,
        -1,
        -1,
        -1,
        -1,
        1
    );
    *outputlen = outSize + LZMA_PROPS_SIZE;
    return ret;
}


__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_lzmaCompressByteArray
(JNIEnv* env,
jclass class,
jbyteArray input,
jint inputlen,
jbyteArray output,
jint outputlen,
int level)
{
    (void) class;
    jint ret = 0;
    jbyte *inputArray, *outputArray = NULL;
    inputArray = (*env)->GetPrimitiveArrayCritical(env, input, 0);
    if (!inputArray) {
        ret = -3;
        goto exit;
    }
    outputArray = (*env)->GetPrimitiveArrayCritical(env, output, 0);
    if (!outputArray) {
        ret = -4;
        goto exit;
    }
    int err = LZMA_compress((unsigned char*)outputArray, &outputlen,
        (unsigned char*)inputArray, inputlen, level);
    if (err != SZ_OK) {
        if (err > 0) {
            ret = -err;
        } else {
            ret = err;
        }
    } else {
        ret = outputlen;
    }

exit:
    if (inputArray) {
        (*env)->ReleasePrimitiveArrayCritical(
            env,
            input,
            inputArray,
            JNI_ABORT);
    }
    if (outputArray) {
        (*env)->ReleasePrimitiveArrayCritical(
            env,
            output,
            outputArray,
            ret > 0 ? 0 : JNI_ABORT);
    }
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL JavaCritical_ru_yandex_compress_NativeCompress_lzmaCompressRawArray
(jlong input,
jint inputlen,
jlong output,
jint outputlen,
jint level)
{
    jint ret = 0;
    int err = LZMA_compress((unsigned char*)output, &outputlen,
        (unsigned char*)input, inputlen, level);
    if (err != SZ_OK) {
        if (err > 0) {
            ret = -err;
        } else {
            ret = err;
        }
    } else {
        ret = outputlen;
    }
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_lzmaCompressRawArray
(JNIEnv* env,
jclass class,
jlong input,
jint inputlen,
jlong output,
jint outputlen,
jint level)
{
    (void) class;
    (void) env;
    return JavaCritical_ru_yandex_compress_NativeCompress_lzmaCompressRawArray(
        input,
        inputlen,
        output,
        outputlen,
        level);
}

int LZMA_decompress(
    unsigned char* output,
    size_t outSize,
    unsigned char* input,
    size_t inputSize)
{
    const unsigned char *props = input;
    size_t compressedSize = inputSize - LZMA_PROPS_SIZE;
    int err = LzmaUncompress(
        output,
        &outSize,
        input + LZMA_PROPS_SIZE,
        &compressedSize,
        props,
        LZMA_PROPS_SIZE);
    return err;
}

jint lzmaReadAndDecompressByteArray
(jint fd,
jlong fPos,
jint deflatedlen,
jlong output,
jint outputlen)
{
    void *readBuffer;

    readBuffer = alloca(deflatedlen);
    int read = PreadFull(fd, readBuffer, deflatedlen, fPos);
    if (read == 0 || read < deflatedlen) {
        return EOF_ERROR;
    }
    if (read == -1) {
        return -errno;
    }
    jint ret = LZMA_decompress(
        (unsigned char*)output,
        outputlen,
        (unsigned char*)readBuffer,
        deflatedlen);
    if (ret != SZ_OK) {
        if (ret > 0) {
            ret = -ret;
        }
    }
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_lzmaReadAndDecompressByteArray
(JNIEnv* env,
jclass class,
jint fd,
jlong fPos,
jint deflatedlen,
jlong output,
jint outputlen)
{
    (void) env;
    (void) class;
    return lzmaReadAndDecompressByteArray(fd, fPos, deflatedlen, output, outputlen);
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL JavaCritical_ru_yandex_compress_NativeCompress_lzmaReadAndDecompressByteArray
(jint fd,
jlong fPos,
jint deflatedlen,
jlong output,
jint outputlen)
{
    return lzmaReadAndDecompressByteArray(fd, fPos, deflatedlen, output, outputlen);
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_lzmaDecompressByteArray
(JNIEnv* env,
jclass class,
jbyteArray input,
jint inputlen,
jlong output,
jint outputlen)
{
    (void) class;
    jbyte *array = (*env)->GetPrimitiveArrayCritical(env, input, 0);
    if (!array) {
        return -1;
    }
    jint ret = LZMA_decompress(
        (unsigned char*)output,
        outputlen,
        (unsigned char*)array,
        inputlen);
    (*env)->ReleasePrimitiveArrayCritical(
        env,
        input,
        array,
        JNI_ABORT);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_compress_NativeCompress_lzmaDecompressBytes
(JNIEnv* env,
jclass class,
jlong input,
jint inputlen,
jlong output,
jint outputlen)
{
    (void) class;
    (void) env;
    jint ret = LZMA_decompress(
        (unsigned char*)output,
        outputlen,
        (unsigned char*)input,
        inputlen);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL JavaCritical_ru_yandex_compress_NativeCompress_lzmaDecompressBytes
(jlong input,
jint inputlen,
jlong output,
jint outputlen)
{
    jint ret = LZMA_decompress(
        (unsigned char*)output,
        outputlen,
        (unsigned char*)input,
        inputlen);
    return ret;
}

const char *LZMA_getErrorName(int code) {
    if (code < 0) {
        code = -code;
    }
    switch (code) {
        case SZ_OK:
            return "SZ_OK";
        case SZ_ERROR_DATA:
            return "SZ_ERROR_DATA";
        case SZ_ERROR_MEM:
            return "SZ_ERROR_MEM";
        case SZ_ERROR_UNSUPPORTED:
            return "SZ_ERROR_UNSUPPORTED";
        case SZ_ERROR_INPUT_EOF:
            return "SZ_ERROR_INPUT_EOF";
        default:
            return "SZ_Unhandled error";
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jstring JNICALL Java_ru_yandex_compress_NativeCompress_lzmaCodeToString
(JNIEnv* env,
jclass class,
jint code)
{
    (void) class;
    return (*env)->NewStringUTF(env, LZMA_getErrorName(code));
}

