package ru.yandex.concurrent;

import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

public class FutureBase<T> implements Future<T> {
    protected volatile boolean completed = false;
    protected volatile boolean cancelled = false;
    protected boolean done = false;
    protected T result = null;
    protected Exception e = null;

    protected boolean onCancel(final boolean mayInterruptIfRunning) {
        return true;
    }

    protected void onComplete(final T result) {
    }

    protected void onFailure(final Exception e) {
    }

    // Damned type erasure
    protected boolean completedInternal(final T result) {
        synchronized (this) {
            if (done) {
                return false;
            }
            done = true;
        }
        this.result = result;
        onComplete(result);
        synchronized (this) {
            completed = true;
            notifyAll();
        }
        return true;
    }

    public void failed(final Exception e) {
        synchronized (this) {
            if (done) {
                return;
            }
            done = true;
        }
        this.e = e;
        onFailure(e);
        synchronized (this) {
            completed = true;
            notifyAll();
        }
    }

    public void cancelled() {
    }

    // Future<T> implementation
    @Override
    public boolean cancel(final boolean mayInterruptIfRunning) {
        synchronized (this) {
            if (done) {
                return cancelled;
            }
            done = true;
        }
        boolean result = onCancel(mayInterruptIfRunning);
        synchronized (this) {
            cancelled = true;
            completed = true;
            notifyAll();
        }
        return result;
    }

    private T getResult() throws ExecutionException {
        if (cancelled) {
            throw new CancellationException();
        } else if (e == null) {
            return result;
        } else {
            throw new ExecutionException(e);
        }
    }

    @Override
    public synchronized T get()
        throws ExecutionException, InterruptedException
    {
        while (!completed) {
            wait();
        }
        return getResult();
    }

    @Override
    public synchronized T get(final long timeout, final TimeUnit unit)
        throws ExecutionException, InterruptedException, TimeoutException
    {
        long deadline = System.currentTimeMillis() + unit.toMillis(timeout);
        while (true) {
            if (completed) {
                return getResult();
            } else {
                long now = System.currentTimeMillis();
                if (now >= deadline) {
                    throw new TimeoutException();
                } else {
                    wait(deadline - now);
                }
            }
        }
    }

    @Override
    public boolean isCancelled() {
        return cancelled;
    }

    @Override
    public boolean isDone() {
        return completed;
    }
}

