package ru.yandex.concurrent;

import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.LockSupport;
import java.util.concurrent.locks.ReentrantLock;

import org.jctools.queues.MpmcArrayQueue;

public class LifoWaitBlockingQueue<E> implements BlockingQueue<E> {
    private final int size;
    private final Queue<E> queue;
    private final ReentrantLock lock;
    private final Condition notFull;
    private ConcurrentLinkedDeque<Thread> waiters;

    public LifoWaitBlockingQueue(final int size) {
        this(size, new MpmcArrayQueue<>(size));
    }

    public LifoWaitBlockingQueue(final int size, final Queue<E> queue) {
        this.size = size;
        this.queue = queue;
        lock = new ReentrantLock(false);
        notFull = lock.newCondition();
        waiters = new ConcurrentLinkedDeque<>();
    }

    @Override
    public int drainTo(final Collection<? super E> c) {
        return drainTo(c, Integer.MAX_VALUE);
    }

    @Override
    public int drainTo(final Collection<? super E> c, final int max) {
        int drained = 0;
        E e;
        while (drained < max) {
            e = queue.poll();
            if (e == null) {
                break;
            }
            drained++;
            c.add(e);
        }
        return drained;
    }

    @Override
    public boolean contains(final Object o) {
        return queue.contains(o);
    }

    @Override
    public E remove() {
        final E e = poll();
        if (e == null) {
            throw new NoSuchElementException();
        }
        return e;
    }

    @Override
    public boolean remove(final Object o) {
        final boolean removed = queue.remove(o);
        if (removed) {
            lock.lock();
            try {
                notFull.signalAll();
            } finally {
                lock.unlock();
            }
        }
        return removed;
    }

    @Override
    public int remainingCapacity() {
        return size - queue.size();
    }

    @Override
    public boolean addAll(final Collection<? extends E> c) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsAll(final Collection<?> c) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean retainAll(final Collection<?> c) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean removeAll(final Collection<?> c) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void clear() {
        queue.clear();
        lock.lock();
        try {
            notFull.signalAll();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public E poll() {
        final E e = queue.poll();
        if (e != null) {
            lock.lock();
            try {
                notFull.signal();
            } finally {
                lock.unlock();
            }
        }
        return e;
    }

    @Override
    public E poll(final long timeout, final TimeUnit unit)
        throws InterruptedException
    {
        E e = poll();
        if (e == null) {
            final Thread currentThread = Thread.currentThread();
            waiters.addFirst(currentThread);
            LockSupport.parkNanos(unit.toNanos(timeout));
            waiters.removeFirstOccurrence(currentThread);
            e = poll();
        }
        return e;
    }

    @Override
    public E take() throws InterruptedException {
        E e = poll();
        if (e != null) {
            return e;
        }
        final Thread currentThread = Thread.currentThread();
        while ((e = poll()) == null) {
            waiters.addFirst(currentThread);
            //doppelt check
            if (queue.peek() != null) {
                waiters.removeFirstOccurrence(currentThread);
                continue;
            }
            LockSupport.park();
            waiters.removeFirstOccurrence(currentThread);
            if (Thread.interrupted()) {
                throw new InterruptedException();
            }
        }
        return e;
    }

    @Override
    public boolean offer(final E e) {
        boolean offered = queue.offer(e);
        if (offered) {
            final Thread t = waiters.pollFirst();
            if (t != null) {
                LockSupport.unpark(t);
            }
        }
        return offered;
    }

    @Override
    public boolean offer(final E e, final long timeout, final TimeUnit unit)
        throws InterruptedException
    {
        final boolean success;
        if (offer(e)) {
            success = true;
        } else {
            lock.lock();
            try {
                if (notFull.await(timeout, unit)) {
                    success = offer(e);
                } else {
                    success = false;
                }
            } finally {
                lock.unlock();
            }
        }
        return success;
    }

    @Override
    public void put(final E e) throws InterruptedException {
        if (offer(e)) {
            return;
        }
        lock.lock();
        try {
            while (!offer(e)) {
                notFull.await();
            }
        } finally {
            lock.unlock();
        }
    }

    @Override
    public boolean add(final E e) {
        if (!offer(e)) {
            throw new IllegalStateException("Queue is full");
        }
        return true;
    }

    @Override
    public E peek() {
        return queue.peek();
    }

    @Override
    public E element() {
        E e = queue.peek();
        if (e == null) {
            throw new NoSuchElementException();
        }
        return e;
    }

    @Override
    public Object[] toArray() {
        throw new UnsupportedOperationException();
    }

    @Override
    public <T> T[] toArray(final T[] a) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Iterator<E> iterator() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean isEmpty() {
        return queue.isEmpty();
    }

    @Override
    public int size() {
        return queue.size();
    }

    public static void test(final int consumerCount, final int producerCount)
        throws Exception
    {
        //test
        final int cycleCount = 1000000;
        final int finish = Integer.MIN_VALUE;
        System.out.println("Testing consumer count: " + consumerCount);
        System.out.println("Testing producer count: " + producerCount);
        final LifoWaitBlockingQueue<Integer> queue =
            new LifoWaitBlockingQueue<>(consumerCount);
        Thread[] producers = new Thread[producerCount];
        for (int i = 0; i < producerCount; i++) {
            final int n = i;
            producers[i] = new Thread("Producer-" + n) {
                @Override
                public void run() {
                    try {
                        long start = System.currentTimeMillis();
                        for (int i = 0; i < cycleCount; i++) {
                            queue.put(i);
//                            if (i % printInterval == 0) {
//                                System.out.println("produced: " + i);
//                            }
                        }
                        System.out.println("Producer<" + n + "> Finished in: "
                            + (System.currentTimeMillis() - start));
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            };
        }
        Thread[] consumers = new Thread[consumerCount];
        for (int i = 0; i < consumerCount; i++) {
            final int n = i;
            consumers[i] = new Thread("Consumer-" + n) {
                @Override
                public void run() {
                    try {
                        long start = System.currentTimeMillis();
                        int i;
                        do {
                            i = queue.take();
//                            if (i % printInterval == 0) {
//                                System.out.println("consumer: " + i);
//                            }
                        } while (i != finish);
                        System.out.println(
                            "Consumer<" + n + "> finished in: "
                                + (System.currentTimeMillis() - start));
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            };
            consumers[i].start();
        }
        for (int i = 0; i < producerCount; i++) {
            producers[i].start();
        }
        for (int i = 0; i < producerCount; i++) {
            producers[i].join();
        }
        for (int i = 0; i < consumerCount; i++) {
            queue.put(finish);
        }
        for (int i = 0; i < consumerCount; i++) {
            while (consumers[i].isAlive()) {
                consumers[i].join(TimeUnit.SECONDS.toMillis(1));
                System.out.println("peek: " + queue.peek() + '/'
                    + queue.waiters.toString());
            }
        }
    }

    public static void main(final String[] args) throws Exception {
        for (int p = 1; p < (2 + 2); p++) {
            for (int c = 2; c < (2 << 2); c++) {
                test(c, p);
            }
        }
    }
}
