package ru.yandex.concurrent;

import java.io.Serializable;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class LockStorage<K, V> implements Serializable {
    private static final long serialVersionUID = 0L;

    private static class Entry<V> implements Serializable {
        private static final long serialVersionUID = 0L;

        private final V value;
        private int refs = 1;

        Entry(final V value) {
            this.value = value;
        }
    }

    private final ConcurrentHashMap<K, Entry<V>> storage =
        new ConcurrentHashMap<>();

    public V acquire(final K key, final V value) {
        return acquire(key, new Entry<>(value));
    }

    private V acquire(final K key, final Entry<V> entry) {
        Entry<V> oldEntry = storage.putIfAbsent(key, entry);
        V result;
        if (oldEntry == null) {
            result = entry.value;
        } else {
            synchronized (oldEntry) {
                if (oldEntry.refs > 0) {
                    ++oldEntry.refs;
                    return oldEntry.value;
                }
            }
            result = acquire(key, entry);
        }
        return result;
    }

    public int release(final Object key) {
        int refs;
        Entry<V> entry = storage.get(key);
        synchronized (entry) {
            refs = --entry.refs;
            if (refs == 0) {
                storage.remove(key);
            }
        }
        return refs;
    }

    public int size() {
        return storage.size();
    }

    public <E extends Exception> void traverse(
        final Visitor<K, V, ? extends E> visitor)
        throws E
    {
        for (Map.Entry<K, Entry<V>> entry: storage.entrySet()) {
            K key = entry.getKey();
            Entry<V> storageEntry = entry.getValue();
            synchronized (storageEntry) {
                int refs = storageEntry.refs;
                if (refs > 0) {
                    visitor.visit(key, refs, storageEntry.value);
                }
            }
        }
    }

    public interface Visitor<K, V, E extends Exception> {
        void visit(K key, int refs, V value) throws E;
    }
}

