package ru.yandex.concurrent;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Consumer;

import ru.yandex.util.timesource.TimeSource;

public class TimeFrameQueue<T> implements Consumer<T>, Iterable<T> {
    private final Queue<Queue<Node<T>>> queues = new ConcurrentLinkedQueue<>();

    @SuppressWarnings("ThreadLocalUsage")
    private final ThreadLocalQueue<T> queue = new ThreadLocalQueue<>(queues);
    private long maxAge;

    public TimeFrameQueue(final long maxAge) {
        this.maxAge = maxAge;
    }

    @Override
    public void accept(final T t) {
        Queue<Node<T>> queue = this.queue.get();
        long now = TimeSource.INSTANCE.currentTimeMillis();
        while (true) {
            Node<T> node = queue.peek();
            if (node == null || node.deadline() > now) {
                break;
            }
            queue.poll();
        }
        queue.add(new Node<>(now + maxAge, t));
    }

    @Override
    public Iterator<T> iterator() {
        return new Itr<T>(queues);
    }

    private static class Node<T> {
        private final long deadline;
        private final T t;

        Node(final long deadline, final T t) {
            this.deadline = deadline;
            this.t = t;
        }

        public long deadline() {
            return deadline;
        }

        public T get() {
            return t;
        }
    }

    private static class ThreadLocalQueue<T>
        extends ThreadLocal<Queue<Node<T>>>
    {
        private final Queue<Queue<Node<T>>> queues;

        ThreadLocalQueue(final Queue<Queue<Node<T>>> queues) {
            this.queues = queues;
        }

        @Override
        public Queue<Node<T>> initialValue() {
            Queue<Node<T>> queue = new ConcurrentLinkedQueue<>();
            queues.add(queue);
            return queue;
        }
    }

    private static class Itr<T> implements Iterator<T> {
        private final long now = TimeSource.INSTANCE.currentTimeMillis();
        private final Iterator<Queue<Node<T>>> queuesIter;
        private Iterator<Node<T>> currentQueueIter = null;
        private T nextItem = null;

        Itr(final Queue<Queue<Node<T>>> queues) {
            queuesIter = queues.iterator();
            init();
        }

        private void init() {
            while (queuesIter.hasNext()) {
                currentQueueIter = queuesIter.next().iterator();
                while (currentQueueIter.hasNext()) {
                    Node<T> node = currentQueueIter.next();
                    if (node.deadline() > now) {
                        nextItem = node.get();
                        return;
                    }
                }
            }
        }

        private T advance() {
            while (true) {
                while (currentQueueIter.hasNext()) {
                    Node<T> node = currentQueueIter.next();
                    if (node.deadline() > now) {
                        return node.get();
                    }
                }
                if (queuesIter.hasNext()) {
                    currentQueueIter = queuesIter.next().iterator();
                } else {
                    return null;
                }
            }
        }

        @Override
        public boolean hasNext() {
            return nextItem != null;
        }

        @Override
        public T next() {
            if (nextItem == null) {
                throw new NoSuchElementException();
            }
            T result = nextItem;
            nextItem = advance();
            return result;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}

