package ru.yandex.parser.config;

import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import ru.yandex.collection.SingletonIterator;
import ru.yandex.function.StringBuilderable;
import ru.yandex.function.StringProcessor;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.ValuesStorage;
import ru.yandex.util.string.StringUtils;

public class IniConfig
    implements ValuesStorage<ConfigException>, StringBuilderable
{
    private static final CollectionParser<
        String,
        List<String>,
        RuntimeException> PATH_PARSER =
            new CollectionParser<>(x -> x, ArrayList::new, '.');

    private final StringProcessor<List<String>, RuntimeException> pathParser =
        new StringProcessor<>(PATH_PARSER);
    private final Map<String, String> keys = new LinkedHashMap<>();
    private final Set<String> unusedKeys = new HashSet<>();
    private final Map<String, IniConfig> sections = new LinkedHashMap<>();
    private final Path path;
    private final String prefix;
    private boolean used = false;

    public IniConfig(final File file) throws ConfigException, IOException {
        this(file.toPath());
    }

    public IniConfig(final Path path) throws ConfigException, IOException {
        this.path = path;
        prefix = "";
        try (Reader reader = new InputStreamReader(
                Files.newInputStream(path),
                StandardCharsets.UTF_8.newDecoder()
                    .onMalformedInput(CodingErrorAction.REPORT)
                    .onUnmappableCharacter(CodingErrorAction.REPORT)))
        {
            new IniLoader(this, path, reader);
        }
    }

    public IniConfig(final Reader reader) throws ConfigException, IOException {
        this.path = null;
        prefix = "";
        new IniLoader(this, reader);
    }

    public IniConfig(final Path path, final Reader reader)
        throws ConfigException, IOException
    {
        this.path = path;
        prefix = "";
        new IniLoader(this, path, reader);
    }

    private IniConfig(final String prefix) {
        this.prefix = prefix;
        path = null;
    }

    public void put(final String key, final String value) {
        List<String> path = pathParser.process(key);
        IniConfig section = this;
        used = true;
        for (int i = 0; i < path.size() - 1; ++i) {
            section = section.section(path.get(i));
        }
        String last = path.get(path.size() - 1);
        section.used = true;
        section.unusedKeys.add(last);
        section.keys.put(last, value);
    }

    public void putAsString(final String key, final Object value) {
        if (value != null) {
            put(key, value.toString());
        }
    }

    public void putAsString(final String key, final Collection<?> values) {
        if (values != null && !values.isEmpty()) {
            put(
                key,
                StringUtils.join(
                    values,
                    (sb, x) -> sb.append(x),
                    ", "));
        }
    }

    public <E extends Enum<E>> void putEnum(final String key, final E value) {
        if (value != null) {
            put(
                key,
                value.toString().toLowerCase(Locale.ROOT).replace('_', '-'));
        }
    }

    public void removeKey(final String key) {
        List<String> path = pathParser.process(key);
        IniConfig section = this;
        used = true;
        for (int i = 0; i < path.size() - 1; ++i) {
            section = section.section(path.get(i));
        }
        String last = path.get(path.size() - 1);
        section.unusedKeys.remove(last);
        section.keys.remove(last);
    }

    public IniConfig sectionOrDefault(
        final String section,
        final IniConfig defaultSection)
    {
        IniConfig current = sectionOrNull(section);
        if (current == null) {
            return defaultSection;
        }

        return current;
    }

    public IniConfig sectionOrNull(final String section) {
        used = true;
        IniConfig current = sections.get(section);
        if (current == null) {
            List<String> path = pathParser.process(section);
            current = this;
            for (String name: path) {
                current = current.sections.get(name);
                if (current == null) {
                    break;
                }
            }
        }
        return current;
    }

    public IniConfig section(final String section) {
        used = true;
        IniConfig current = sections.get(section);
        if (current == null) {
            List<String> path = pathParser.process(section);
            current = this;
            for (String name: path) {
                IniConfig subsection = current.sections.get(name);
                if (subsection == null) {
                    subsection =
                        new IniConfig(current.prefix + name + '.');
                    current.sections.put(name, subsection);
                }
                current.used = true;
                current = subsection;
            }
        }
        return current;
    }

    public Set<String> keys() {
        used = true;
        return keys.keySet();
    }

    public Map<String, IniConfig> sections() {
        used = true;
        return sections;
    }

    public List<String> unusedKeys() {
        List<String> unusedKeys = new ArrayList<>(this.unusedKeys);
        boolean used = this.used;
        for (Map.Entry<String, IniConfig> entry: sections.entrySet()) {
            for (String key: entry.getValue().unusedKeys()) {
                unusedKeys.add(entry.getKey() + '.' + key);
            }
        }
        if (!used) {
            unusedKeys.add("[]");
        }
        return unusedKeys;
    }

    public void checkUnusedKeys() throws ConfigException {
        List<String> unusedKeys = unusedKeys();
        if (!unusedKeys.isEmpty()) {
            StringBuilder sb = new StringBuilder(
                "The following configuration keys wasn't used: ");
            sb.append(unusedKeys);
            if (path != null) {
                sb.append(", while parsing ");
                sb.append(path);
            }
            throw new ConfigException(new String(sb));
        }
    }

    public void renameSection(final String from, final String to)
        throws ConfigException
    {
        if (sections.containsKey(to)) {
            throw new ConfigException(
                "Renaming '" + from + "' will overwrite existing section '"
                + to + '\'');
        }
        IniConfig section = sections.remove(from);
        if (section == null) {
            sections.remove(to);
        } else {
            sections.put(to, section);
        }
    }

    @Override
    public String getOrNull(final String key) {
        used = true;
        String value = keys.get(key);
        if (value == null) {
            List<String> path = pathParser.process(key);
            IniConfig section = this;
            for (int i = 0; i < path.size() - 1 && section != null; ++i) {
                section = section.sections.get(path.get(i));
            }
            if (section != null) {
                String last = path.get(path.size() - 1);
                section.unusedKeys.remove(last);
                section.used = true;
                value = section.keys.get(last);
            }
        } else {
            unusedKeys.remove(key);
        }
        return value;
    }

    @Override
    public String getLastOrNull(final String name) {
        // Exactly one value per key
        return getOrNull(name);
    }

    @Override
    public Iterator<String> getAllOrNull(final String name) {
        String value = getOrNull(name);
        if (value == null) {
            return null;
        } else {
            return new SingletonIterator<>(value);
        }
    }

    @Override
    public ParameterNotSetException parameterNotSetException(
        final String name)
    {
        return new ParameterNotSetException(prefix + name);
    }

    @Override
    public ConfigException parseFailedException(
        final String name,
        final String value,
        final Throwable cause)
    {
        return new ConfigException(
            "Failed to parse parameter " + prefix + name
            + " with value '" + value + '\'',
            cause);
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        toStringBuilder(sb, "");
    }

    // TODO: support sectionName, key and value special characters escaping
    public void toStringBuilder(
        final StringBuilder sb,
        final String sectionName)
    {
        if (!keys.isEmpty()) {
            if (!sectionName.isEmpty()) {
                sb.append('[');
                sb.append(sectionName);
                sb.append(']');
                sb.append('\n');
            }
            for (Map.Entry<String, String> key: keys.entrySet()) {
                sb.append(key.getKey());
                sb.append(" = ");
                sb.append(key.getValue());
                sb.append('\n');
            }
            sb.append('\n');
        }

        String nextSectionPrefix;
        if (sectionName.isEmpty()) {
            nextSectionPrefix = "";
        } else {
            nextSectionPrefix = StringUtils.concat(sectionName, '.');
        }

        for (Map.Entry<String, IniConfig> section: sections.entrySet()) {
            section.getValue().toStringBuilder(
                sb,
                StringUtils.concat(nextSectionPrefix, section.getKey()));
        }
    }

    @Override
    public String toString() {
        return toStringFast();
    }

    public String prefix() {
        return prefix;
    }

    // Parses config and prints requested parameter value
    public static void main(final String... args)
        throws ConfigException, IOException
    {
        if (args.length < 2) {
            System.err.println(
                "Usage: " + IniConfig.class.getName()
                + " <config file> <parameter name>");
            System.exit(1);
        }
        System.out.println(
            new IniConfig(new File(args[0])).getString(args[1]));
    }

    public static IniConfig empty() {
        return new IniConfig("");
    }
}

