package ru.yandex.parser.config;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.util.filesystem.PathUtils;
import ru.yandex.util.string.StringUtils;

public class IniLoader {
    private static final Pattern EXT_PATTERN =
        Pattern.compile("\\.(?=[^\\.]+$)");
    private static final Pattern SECTION = Pattern.compile("^\\s*\\[");
    private static final Pattern INCLUDE =
        Pattern.compile("^\\s*[$][(]include(?<subsection>_subsection)?\\s+");;
    private static final String NUMBER_OR_SUBST = "0123456789$";

    private final IniConfig root;
    private final String fileName;
    private final Path configDir;
    private IniConfig current;
    private ConfigException exception = null;

    public IniLoader(final IniConfig root, final Reader config)
        throws ConfigException, IOException
    {
        this.root = root;
        fileName = null;
        configDir = null;
        loadConfig(config);
    }

    @SuppressWarnings("StringSplitter")
    public IniLoader(
        final IniConfig root,
        final Path config,
        final Reader reader)
        throws ConfigException, IOException
    {
        this.root = root;
        fileName =
            EXT_PATTERN.split(Objects.toString(config.getFileName(), ""))[0];
        configDir = config.toAbsolutePath().getParent();
        loadConfig(reader);
        Path overrides =
            config.resolveSibling(config.getFileName() + ".overrides");
        if (Files.exists(overrides)) {
            try (Reader overridesReader = new InputStreamReader(
                    Files.newInputStream(overrides),
                    StandardCharsets.UTF_8.newDecoder()
                        .onMalformedInput(CodingErrorAction.REPORT)
                        .onUnmappableCharacter(CodingErrorAction.REPORT)))
            {
                loadConfig(overridesReader);
            }
        }
    }

    private void loadConfig(final Reader reader)
        throws ConfigException, IOException
    {
        current = root;
        Properties loader = new Properties() {
            private static final long serialVersionUID = 0L;

            @Override
            @SuppressWarnings("UnsynchronizedOverridesSynchronized")
            public Object put(final Object key, final Object value) {
                current.put(key.toString(), expandValue(value.toString()));
                return null;
            }
        };
        StringBuilder sb = new StringBuilder();
        try (BufferedReader lineReader = new BufferedReader(reader)) {
            while (true) {
                String line = lineReader.readLine();
                if (line == null) {
                    loader.load(new StringReader(sb.toString()));
                    break;
                }
                Matcher matcher = SECTION.matcher(line);
                if (matcher.find() && line.endsWith("]")) {
                    loader.load(new StringReader(sb.toString()));
                    sb.setLength(0);
                    processSectionBegin(
                        line.substring(matcher.end(), line.length() - 1));
                } else {
                    matcher = INCLUDE.matcher(line);
                    if (matcher.find() && line.endsWith(")")) {
                        loader.load(new StringReader(sb.toString()));
                        sb.setLength(0);
                        IniConfig parent;
                        if (matcher.group("subsection") == null) {
                            parent = root;
                        } else {
                            parent = current;
                        }
                        processInclude(line, matcher.end(), parent);
                    } else {
                        sb.append(line);
                        sb.append('\n');
                    }
                }
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    private void accountException(final ConfigException exception) {
        if (this.exception == null) {
            this.exception = exception;
        } else {
            this.exception.addSuppressed(exception);
        }
    }

    private String trimValue(final String value) {
        final String rawValue;
        if (isProperty(value)) {
            rawValue = value.substring(2, value.length() - 1).trim();
        } else {
            rawValue = value.trim();
        }
        return rawValue.replaceAll("[ \t]*?", "");
    }

    private long longValue(final String value) {
        try {
            return Long.parseLong(value);
        } catch (NumberFormatException e) {
            accountException(
                new ConfigException(
                    "Invalid number value "
                    + value + " in property expresion"));
            return 0;
        }
    }

    private String add(final String arg1, final String arg2) {
        return Long.toString(
            longValue(propertyValue(arg1))
                + longValue(propertyValue(arg2)));
    }

    private String substract(final String arg1, final String arg2) {
        return Long.toString(
            longValue(propertyValue(arg1))
                - longValue(propertyValue(arg2)));
    }

    private String mul(final String arg1, final String arg2) {
        return Long.toString(
            longValue(propertyValue(arg1))
                * longValue(propertyValue(arg2)));
    }

    private String div(final String arg1, final String arg2) {
        return Long.toString(
            longValue(propertyValue(arg1))
                / longValue(propertyValue(arg2)));
    }

    private String remainder(final String arg1, final String arg2) {
        return Long.toString(
            longValue(propertyValue(arg1))
                % longValue(propertyValue(arg2)));
    }

    private boolean isProperty(final String value) {
        return value.charAt(0) == '$' && value.charAt(1) == '('
            && value.charAt(value.length() - 1) == ')';
    }

    private String getValue(final String key) {
        return getValue(key, true);
    }

    private String getValue(final String key, final boolean accountException) {
        String value;
        if ("dirname".equals(key)) {
            if (configDir == null) {
                value = null;
                if (accountException) {
                    accountException(
                        new ConfigException(
                            "'dirname' directive is not supported for Reader"));
                }
            } else {
                value = configDir.toString();
            }
        } else {
            value = current.getOrNull(key);
            if (value == null) {
                value = root.getOrNull(key);
                if (value == null) {
                    value = System.getProperty(key);
                    if (value == null) {
                        value = System.getenv(key);
                        if (value == null) {
                            if (key.equals("filename")) {
                                value = fileName;
                            }
                            if (value == null && accountException) {
                                accountException(
                                    new ConfigException(
                                        "No property with name <"
                                        + key + "> exists"));
                            }
                        }
                    }
                }
            }
        }
        return value;
    }

    // CSOFF: ReturnCount
    @SuppressWarnings("StringSplitter")
    private String propertyValue(final String value) {
        if (!isProperty(value)) {
            return value;
        }
        final String rawValue = trimValue(value);
        final String[] tokens = rawValue.split("[\\+\\-\\*\\/%]");
        if (tokens.length != 2) {
            return getValue(rawValue);
        }
        final char operator = rawValue.charAt(tokens[0].length());
        final String arg1 = tokens[0].trim();
        final String arg2 = tokens[1].trim();
        if (arg1.isEmpty() || arg2.isEmpty()
            || NUMBER_OR_SUBST.indexOf(arg1.charAt(0)) == -1
            || NUMBER_OR_SUBST.indexOf(arg2.charAt(0)) == -1)
        {
            return getValue(rawValue);
        }
        switch (operator) {
            case '+':
                return add(arg1, arg2);
            case '-':
                return substract(arg1, arg2);
            case '/':
                return div(arg1, arg2);
            case '%':
                return remainder(arg1, arg2);
            case '*':
                return mul(arg1, arg2);
            default:
                accountException(
                    new ConfigException(
                        "Unknown operator: '" + operator
                         + "' in system property token: " + value));
                return null;
        }
    }
    // CSON: ReturnCount

    private String findNextProperty(final String str) {
        int start = str.indexOf("$(");
        if (start != -1) {
            int encloses = 0;
            String expr = null;
            for (int i = start + 2; i < str.length(); i++) {
                final char c = str.charAt(i);
                if (c == '(') {
                    encloses++;
                } else if (c == ')') {
                    if (encloses == 0) {
                        expr = str.substring(start, i + 1);
                        break;
                    }
                    encloses--;
                }
            }
            if (encloses > 0 || expr == null) {
                accountException(
                    new ConfigException(
                        "Unclosed property expression: " + str));
            }
            return expr;
        } else {
            return null;
        }
    }

    private String expandValue(final String value) {
        //replaces each occurrence of $(...) with the value of
        //property expression
        //
        //examples:
        //  System.getProperty("httpport") = 8080
        //  System.getProperty("root") = "/u0"
        //  System.getProperty("logpath") = "logs"
        //
        //ex1:
        //  port = $(httport)
        //  result: port = System.getProperty("httpport") = 8080
        //
        //ex2:
        //  port = $($(httpport) + 1)
        //  result: port = System.getProperty("httpport") + 1 = 8081
        //
        //ex3:
        //  logpath = $(root)/$(logs)/lucene
        //  result: logpath = /u0/logs/lucene
        String parsedValue = value;
        String property;
        while ((property = findNextProperty(parsedValue)) != null) {
            int start = parsedValue.indexOf(property);
            parsedValue = StringUtils.concat(
                parsedValue.substring(0, start),
                propertyValue(property),
                parsedValue.substring(start + property.length()));
        }
        return parsedValue;
    }

    private void processSectionBegin(final String section) {
        current = root.section(expandValue(section));
    }

    private void processInclude(
        final String line,
        final int prefixLength,
        final IniConfig parentSection)
    {
        if (configDir == null) {
            accountException(
                new ConfigException(
                    "include directives not supported for Reader"));
        } else {
            String glob =
                expandValue(line.substring(prefixLength, line.length() - 1));
            if (glob.isEmpty()) {
                accountException(
                    new ConfigException("Empty include pattern for: " + line));
            } else {
                String configDirs = getValue("CONFIG_DIRS", false);
                List<Path> includeDirs;
                if (configDirs == null) {
                    includeDirs = Collections.singletonList(configDir);
                } else {
                    includeDirs = new ArrayList<>();
                    includeDirs.add(configDir);
                    for (String part: configDirs.split(",")) {
                        includeDirs.add(Paths.get(part.trim()));
                    }
                }

                Set<Path> files = Collections.emptySet();
                for (Path includeDir: includeDirs) {
                    try (Stream<Path> filesStream =
                            PathUtils.glob(
                                includeDir,
                                glob,
                                FileVisitOption.FOLLOW_LINKS))
                    {
                        files =
                            filesStream.collect(
                                Collectors.toCollection(TreeSet::new));
                        if (files.isEmpty()) {
                            Path dir;
                            String cuttedGlob;
                            if (glob.startsWith(File.separator)) {
                                int folderIndex =
                                    glob.lastIndexOf(File.separator);
                                dir =
                                    Paths.get(glob.substring(0, folderIndex));
                                cuttedGlob = glob.substring(folderIndex + 1);
                            } else {
                                int folderIndex =
                                    glob.lastIndexOf(File.separator);
                                if (folderIndex > 0) {
                                    dir =
                                        Paths.get(
                                            glob.substring(0, folderIndex));
                                    cuttedGlob =
                                        glob.substring(folderIndex + 1);
                                } else {
                                    dir = Paths.get("");
                                    cuttedGlob = glob;
                                }
                            }

                            // try absolute path
                            try (Stream<Path> absolutePathFilesStream =
                                    PathUtils.glob(
                                        dir,
                                        cuttedGlob,
                                        FileVisitOption.FOLLOW_LINKS))
                            {
                                    files =
                                        absolutePathFilesStream.collect(
                                            Collectors.toCollection(
                                                TreeSet::new));
                            }
                        }
                        if (!files.isEmpty()) {
                            break;
                        }
                    } catch (IOException e) {
                        accountException(
                            new ConfigException(
                                "Failed to list files under " + includeDir
                                + " with glob: " + glob));
                    }
                }

                if (files.isEmpty()) {
                    accountException(
                        new ConfigException(
                            "No include files found for glob: " + glob
                                + " in directories " + includeDirs
                                + " or in " + Paths.get("")));
                } else {
                    for (Path config: files) {
                        try {
                            try (Reader reader =
                                    new InputStreamReader(
                                        Files.newInputStream(config),
                                        StandardCharsets.UTF_8.newDecoder()
                                            .onMalformedInput(
                                                CodingErrorAction.REPORT)
                                            .onUnmappableCharacter(
                                                CodingErrorAction.REPORT)))
                            {
                                new IniLoader(
                                    parentSection,
                                    config,
                                    reader);
                            }
                        } catch (ConfigException | IOException e) {
                            accountException(
                                new ConfigException(
                                    "Failed to load file: " + config,
                                    e));
                        }
                    }
                }
            }
        }
        current = parentSection;
    }
}

