package ru.yandex.parser.config;

import java.util.Map;
import java.util.Set;
import java.util.function.IntFunction;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.function.GenericFunction;

public class ParameterMapConfig<T> {
    private final String name;
    private final GenericFunction<String, ? extends T, ? extends Exception>
        parser;
    private final GenericFunction<? super T, ? extends T, ? extends Exception>
        validator;
    private final IntFunction<? extends Map<String, T>> mapSupplier;

    public ParameterMapConfig(
        final String name,
        final GenericFunction<String, ? extends T, ? extends Exception> parser,
        final GenericFunction<? super T, ? extends T, ? extends Exception>
        validator,
        final IntFunction<? extends Map<String, T>> mapSupplier)
    {
        this.name = name;
        this.parser = parser.andThen(validator);
        this.validator = validator;
        this.mapSupplier = mapSupplier;
    }

    public Map<String, T> extract(
        final IniConfig config,
        final Map<String, T> defaults)
        throws ConfigException
    {
        IniConfig section = config.sectionOrNull(name);
        if (section == null) {
            return defaults;
        }
        Set<String> keys = section.keys();
        Map<String, T> result = mapSupplier.apply(keys.size());
        for (String key: keys) {
            try {
                result.put(key, section.get(key, parser));
            } catch (Throwable t) {
                throw new ConfigException(
                    "Failed to parse parameter '"
                    + config.prefix() + name + '.' + key + '\'',
                    t);
            }
        }
        return result;
    }

    public Map<String, T> validate(final Map<String, T> value)
        throws ConfigException
    {
        if (value == null) {
            throw new ConfigException("Parameter map " + name + " is not set");
        }
        Map<String, T> result = mapSupplier.apply(value.size());
        for (Map.Entry<String, T> entry: value.entrySet()) {
            String key = entry.getKey();
            try {
                result.put(key, validator.apply(entry.getValue()));
            } catch (Throwable t) {
                throw new ConfigException(
                    "Validation for parameter " + name + '.' + key + " failed",
                    t);
            }
        }
        return CollectionCompactor.compact(result);
    }
}

