package com.twmacinta.util;

import ru.yandex.function.ByteArrayVoidProcessor;

/**
 * Fast implementation of RSA's MD5 hash generator in Java JDK Beta-2 or higher.
 * <p>
 * Originally written by Santeri Paavolainen, Helsinki Finland 1996.<br>
 * (c) Santeri Paavolainen, Helsinki Finland 1996<br>
 * Many changes Copyright (c) 2002 - 2010 Timothy W Macinta<br>
 * <p>
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * <p>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * <p>
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * <p>
 * See http://www.twmacinta.com/myjava/fast_md5.php for more information
 * on this file and the related files.
 * <p>
 * This was originally a rather straight re-implementation of the
 * reference implementation given in RFC1321 by RSA.  It passes the MD5
 * test suite as defined in RFC1321.
 * <p>
 * Many optimizations made by Timothy W Macinta.  Reduced time to checksum a
 * test file in Java alone to roughly half the time taken compared with
 * java.security.MessageDigest (within an intepretter).  Also added an
 * optional native method to reduce the time even further.
 * See http://www.twmacinta.com/myjava/fast_md5.php for further information
 * on the time improvements achieved.
 * <p>
 * Some bug fixes also made by Timothy W Macinta.
 * <p>
 * Please note: I (Timothy Macinta) have put this code in the
 * com.twmacinta.util package only because it came without a package.  I
 * was not the the original author of the code, although I did
 * optimize it (substantially) and fix some bugs.
 * <p>
 * This Java class has been derived from the RSA Data Security, Inc. MD5
 * Message-Digest Algorithm and its reference implementation.
 * <p>
 * This class will attempt to use a native method to quickly compute
 * checksums when the appropriate native library is available.  On Linux,
 * this library should be named "MD5.so" and on Windows it should be
 * named "MD5.dll".  The code will attempt to locate the library in the
 * following locations in the order given:
 *
 * <ol>
 *   <li>The path specified by the system property
 *       "com.twmacinta.util.MD5.NATIVE_LIB_FILE"
 *       (be sure to include "MD5.so", "MD5.dll",
 *       or "MD5.jnilib" as appropriate at the end
 *       of the path).
 *   <li>A platform specific directory beneath the "lib/arch/" directory.
 *       For example, On Windows for 32 bit x86 architectures, this is
 *       "lib/arch/win32_x86/".
 *   <li>Within the "lib/" directory.
 *   <li>Within the current directory.
 * </ol>
 *
 * <p>
 * If the library is not found, the code will fall back to the default
 * (slower) Java code.
 * <p>
 * As a side effect of having the code search for the native library,
 * SecurityExceptions might be thrown on JVMs that have a restrictive
 * SecurityManager.  The initialization code attempts to silently discard
 * these exceptions and continue, but many SecurityManagers will
 * attempt to notify the user directly of all SecurityExceptions thrown.
 * Consequently, the code has provisions for skipping the search for
 * the native library.  Any of these provisions may be used to skip the
 * search as long as they are performed <i>before</i> the first
 * instance of a com.twmacinta.util.MD5 object is constructed (note that
 * the convenience stream objects will implicitly create an MD5 object).
 * <p>
 * The first option is to set the system property
 * "com.twmacinta.util.MD5.NO_NATIVE_LIB" to "true" or "1".
 * Unfortunately, SecurityManagers may also choose to disallow system
 * property setting, so this won't be of use in all cases.
 * <p>
 * The second option is to call
 * com.twmacinta.util.MD5.initNativeLibrary(true) before any MD5 objects
 * are constructed.
 *
 * @author Santeri Paavolainen <sjpaavol@cc.helsinki.fi>
 * @author Timothy W Macinta (twm@alum.mit.edu) (optimizations and bug fixes)
 */

public class MD5 implements ByteArrayVoidProcessor<RuntimeException> {
    static {
        System.loadLibrary("fast-md5");
    }

    /**
     * Padding for Final()
     **/
    private static final byte[] PADDING = {
        (byte) 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
    };

    /**
    * 128-bit state
    */
    private final int[] state = new int[]{0x67452301, 0xefcdab89, 0x98badcfe, 0x10325476};

    /**
    * 64-byte buffer (512 bits) for storing to-be-hashed characters
    */
    private final byte[] buffer = new byte[64];

    /**
    * 64-bit character count
    */
    long count = 0;

    private static native void doTransform(int[] state, byte[] buffer, int shift, int length);

    /**
     * Plain update, updates this object
     **/
    public void Update(byte[] buffer, int length) {
        Update(buffer, 0, length);
    }

    /**
     * Updates hash with the bytebuffer given (using at maximum length bytes from
     * that buffer)
     *
     * @param buffer Array of bytes to be hashed
     * @param offset Offset to buffer array
     * @param length Use at maximum `length' bytes (absolute
     *               maximum is buffer.length)
     */
    public void Update(byte[] buffer, int offset, int length) {
        int index, partlen, i, start;

        /* Length can be told to be shorter, but not inter */
        if ((length - offset)> buffer.length)
            length = buffer.length - offset;

        /* compute number of bytes mod 64 */

        index = (int) (count & 0x3f);
        count += length;

        partlen = 64 - index;

        if (length >= partlen) {

            // update state (using native method) to reflect input

            if (partlen == 64) {
                partlen = 0;
            } else {
                for (i = 0; i < partlen; i++) {
                    this.buffer[i + index] = buffer[i + offset];
                }
                doTransform(state, this.buffer, 0, 64);
            }
            i = partlen + ((length - partlen) / 64) * 64;

            // break into chunks to guard against stack overflow in JNI

            int transformLength = length - partlen;
            int transformOffset = partlen + offset;
            final int maxLength = 65536; // prevent stack overflow in JNI
            while (true) {
                if (transformLength > maxLength) {
                    doTransform(state, buffer, transformOffset, maxLength);
                    transformLength -= maxLength;
                    transformOffset += maxLength;
                } else {
                    doTransform(state, buffer, transformOffset, transformLength);
                    break;
                }
            }
            index = 0;
        } else {
            i = 0;
        }

        /* buffer remaining input */
        if (i < length) {
            start = i;
            for (; i < length; i++) {
                this.buffer[index + i - start] = buffer[i + offset];
            }
        }
    }

    /**
     * Updates hash with given array of bytes
     *
     * @param buffer Array of bytes to use for updating the hash
     **/
    public void Update(byte[] buffer) {
        Update(buffer, 0, buffer.length);
    }

    /**
     * Updates hash with a single byte
     *
     * @param b Single byte to update the hash
     **/
    public void Update(byte b) {
        byte[] buffer = new byte[1];
        buffer[0] = b;

        Update(buffer, 1);
    }

    /**
     * Update buffer with a single integer (only & 0xff part is used,
     * as a byte)
     *
     * @param i Integer value, which is then converted to byte as i & 0xff
     **/
    public void Update(int i) {
        Update((byte) (i & 0xff));
    }

    private static byte[] Encode(int[] input, int len) {
        int i, j;
        byte[] out;

        out = new byte[len];

        for (i = j = 0; j  < len; i++) {
            out[j++] = (byte) (input[i] & 0xff);
            out[j++] = (byte) ((input[i] >>> 8) & 0xff);
            out[j++] = (byte) ((input[i] >>> 16) & 0xff);
            out[j++] = (byte) ((input[i] >>> 24) & 0xff);
        }

        return out;
    }

    /**
     * Returns array of bytes (16 bytes) representing hash as of the
     * current state of this object. Note: getting a hash does not
     * invalidate the hash object, it only creates a copy of the real
     * state which is finalized.
     *
     * @return Array of 16 bytes, the hash of all updated bytes
     **/
    public byte[] Final() {
        int[] countInts = {(int) (count << 3), (int) (count >> 29)};
        byte[] bits = Encode(countInts, 8);

        int index = (int) (count & 0x3f);
        int padlen = (index < 56) ? (56 - index) : (120 - index);

        Update(PADDING, 0, padlen);
        Update(bits, 0, 8);

        return Encode(state, 16);
    }

    @Override
    public void process(final byte[] buf, final int off, final int len) {
        Update(buf, off, len);
    }
}
