package ru.yandex.net.uri.fast;

import java.net.IDN;
import java.util.Objects;

import ru.yandex.function.StringBuilderable;
import ru.yandex.util.string.StringUtils;

public class FastUri implements StringBuilderable {
    private String scheme;
    private String userInfo;
    private String host;
    private int port;
    private String path;
    private String query;
    private String fragment;
    private String schemeSpecificPart;
    private String authority;

    public FastUri(
        final String scheme,
        final String userInfo,
        final String host,
        final int port,
        final String path,
        final String query,
        final String fragment,
        final String schemeSpecificPart,
        final String authority)
    {
        this.scheme = scheme;
        this.userInfo = userInfo;
        this.host = host;
        this.port = port;
        this.path = path;
        this.query = query;
        this.fragment = fragment;
        this.schemeSpecificPart = schemeSpecificPart;
        this.authority = authority;
    }

    public String scheme() {
        return scheme;
    }

    public FastUri scheme(final String scheme) {
        this.scheme = scheme;
        return this;
    }

    public String userInfo() {
        return userInfo;
    }

    public FastUri userInfo(final String userInfo) {
        this.userInfo = userInfo;
        return this;
    }

    public String host() {
        return host;
    }

    public FastUri host(final String host) {
        this.host = host;
        return this;
    }

    public int port() {
        return port;
    }

    public FastUri port(final int port) {
        this.port = port;
        return this;
    }

    public String path() {
        return path;
    }

    public FastUri path(final String path) {
        this.path = path;
        return this;
    }

    public String query() {
        return query;
    }

    public FastUri query(final String query) {
        this.query = query;
        return this;
    }

    public String fragment() {
        return fragment;
    }

    public FastUri fragment(final String fragment) {
        this.fragment = fragment;
        return this;
    }

    private int expectedAuthorityLength() {
        int len;
        if (host != null) {
            // There is always two slashes before host
            // Don't waste time on needBrackets calculation, just add them
            len = 2 + 2;
            if (userInfo != null) {
                len += userInfo.length();
                ++len;
            }
            len += host.length();
            if (port != -1) {
                // Port will be always positive but we add colon
                len += MAX_SHORT_LENGTH;
            }
        } else if (authority != null) {
            len = authority.length() + 2;
        } else {
            len = 0;
        }
        return len;
    }

    // Assumes that `host != null`
    private boolean needBrackets() {
        return host.indexOf(':') >= 0
            && host.charAt(0) != '[' // very rare case, so check second
            && host.charAt(host.length() - 1) != ']';
    }

    public String authority() {
        return authority;
    }

    public FastUri authority(final String authority) {
        this.authority = authority;
        return this;
    }

    private void appendAuthority(
        final StringBuilder sb,
        final boolean decodePunicode)
    {
        if (host != null) {
            sb.append('/');
            sb.append('/');
            if (userInfo != null) {
                sb.append(userInfo);
                sb.append('@');
            }
            boolean needBrackets = needBrackets();
            if (needBrackets) {
                sb.append('[');
            }
            String host = this.host;
            if (decodePunicode) {
                try {
                    host = IDN.toUnicode(host, IDN.ALLOW_UNASSIGNED);
                } catch (RuntimeException e) {
                    // How this can be?
                }
            }
            sb.append(host);
            if (needBrackets) {
                sb.append(']');
            }
            if (port != -1) {
                sb.append(':');
                sb.append(port);
            }
        } else if (authority != null) {
            sb.append('/');
            sb.append('/');
            sb.append(authority);
        }
    }

    private int portHashCode(final int seed) {
        int p;
        if (port >= 1000) {
            if (port >= 10000) {
                p = 10000;
            } else {
                p = 1000;
            }
        } else if (port >= 100) {
            p = 100;
        } else if (port >= 10) {
            p = 10;
        } else {
            p = 1;
        }

        int hashCode = seed;
        while (p > 0) {
            hashCode = StringUtils.updateHashCode(
                hashCode,
                (char) (((port / p) % 10) + '0'));
            p /= 10;
        }
        return hashCode;
    }

    private int authorityHashCode(final int seed) {
        int hashCode;
        if (host != null) {
            hashCode = StringUtils.updateHashCode(seed, '/');
            hashCode = StringUtils.updateHashCode(hashCode, '/');
            if (userInfo != null) {
                hashCode = StringUtils.updateHashCode(hashCode, userInfo);
                hashCode = StringUtils.updateHashCode(hashCode, '@');
            }
            boolean needBrackets = needBrackets();
            if (needBrackets) {
                hashCode = StringUtils.updateHashCode(hashCode, '[');
            }
            hashCode = StringUtils.updateHashCode(hashCode, host);
            if (needBrackets) {
                hashCode = StringUtils.updateHashCode(hashCode, ']');
            }
            if (port != -1) {
                hashCode = StringUtils.updateHashCode(hashCode, ':');
                hashCode = portHashCode(hashCode);
            }
        } else if (authority == null) {
            hashCode = seed;
        } else {
            hashCode = StringUtils.updateHashCode(seed, '/');
            hashCode = StringUtils.updateHashCode(hashCode, '/');
            hashCode = StringUtils.updateHashCode(hashCode, authority);
        }
        return hashCode;
    }

    private boolean authorityEquals(final FastUri other) {
        if (host == null) {
            return other.host == null
                && Objects.equals(authority, other.authority);
        } else {
            return port == other.port
                && host.equals(other.host)
                && Objects.equals(userInfo, other.userInfo);
        }
    }

    private int expectedSchemeSpecificPartLength() {
        if (schemeSpecificPart == null) {
            int len = expectedAuthorityLength();
            if (path != null) {
                len += path.length();
            }
            if (query != null) {
                ++len;
                len += query.length();
            }
            return len;
        } else {
            return schemeSpecificPart.length();
        }
    }

    public String schemeSpecificPart() {
        return schemeSpecificPart(false);
    }

    private boolean shouldConstructSchemeSpecificPart(
        final boolean decodePunicode)
    {
        return schemeSpecificPart == null
            || (decodePunicode
                && (host != null || authority != null));
    }

    public String schemeSpecificPart(final boolean decodePunicode) {
        if (shouldConstructSchemeSpecificPart(decodePunicode)) {
            StringBuilder sb =
                new StringBuilder(expectedSchemeSpecificPartLength());
            appendSchemeSpecificPart(sb, decodePunicode);
            if (decodePunicode) {
                return new String(sb);
            } else {
                schemeSpecificPart = new String(sb);
            }
        }
        return schemeSpecificPart;
    }

    public FastUri schemeSpecificPart(final String schemeSpecificPart) {
        this.schemeSpecificPart = schemeSpecificPart;
        return this;
    }

    private void appendSchemeSpecificPart(
        final StringBuilder sb,
        final boolean decodePunicode)
    {
        if (shouldConstructSchemeSpecificPart(decodePunicode)) {
            appendAuthority(sb, decodePunicode);
            if (path != null) {
                sb.append(path);
            }
            if (query != null) {
                sb.append('?');
                sb.append(query);
            }
        } else {
            sb.append(schemeSpecificPart);
        }
    }

    @Override
    public int expectedStringLength() {
        int len = 0;
        if (scheme != null) {
            len += scheme.length();
            ++len;
        }
        if (path == null) {
            len += expectedSchemeSpecificPartLength();
        } else {
            len += expectedAuthorityLength();
            len += path.length();
            if (query != null) {
                ++len;
                len += query.length();
            }
        }
        if (fragment != null) {
            ++len;
            len += fragment.length();
        }
        return len;
    }

    @Override
    public String toString() {
        return toString(false);
    }

    public String toString(final boolean decodePunicode) {
        StringBuilder sb = new StringBuilder(expectedStringLength());
        toStringBuilder(sb, decodePunicode);
        return new String(sb);
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        toStringBuilder(sb, false);
    }

    public void toStringBuilder(
        final StringBuilder sb,
        final boolean decodePunicode)
    {
        if (scheme != null) {
            sb.append(scheme);
            sb.append(':');
        }
        if (path == null) {
            appendSchemeSpecificPart(sb, decodePunicode);
        } else {
            appendAuthority(sb, decodePunicode);
            sb.append(path);
            if (query != null) {
                sb.append('?');
                sb.append(query);
            }
        }
        if (fragment != null) {
            sb.append('#');
            sb.append(fragment);
        }
    }

    @Override
    public int hashCode() {
        int hashCode = 0;
        if (scheme != null) {
            hashCode = scheme.hashCode();
            hashCode = StringUtils.updateHashCode(hashCode, ':');
        }
        if (path == null) {
            if (schemeSpecificPart == null) {
                hashCode = authorityHashCode(hashCode);
                if (query != null) {
                    hashCode = StringUtils.updateHashCode(hashCode, '?');
                    hashCode = StringUtils.updateHashCode(hashCode, query);
                }
            } else {
                hashCode =
                    StringUtils.updateHashCode(hashCode, schemeSpecificPart);
            }
        } else {
            hashCode = authorityHashCode(hashCode);
            hashCode = StringUtils.updateHashCode(hashCode, path);
            if (query != null) {
                hashCode = StringUtils.updateHashCode(hashCode, '?');
                hashCode = StringUtils.updateHashCode(hashCode, query);
            }
        }
        if (fragment != null) {
            hashCode = StringUtils.updateHashCode(hashCode, '#');
            hashCode = StringUtils.updateHashCode(hashCode, fragment);
        }
        return hashCode;
    }

    private boolean sspEquals(final FastUri other) {
        return Objects.equals(path, other.path)
            && Objects.equals(query, other.query)
            && authorityEquals(other);
    }

    private boolean equalsInternal(final FastUri other) {
        if (Objects.equals(scheme, other.scheme)
            && Objects.equals(fragment, other.fragment))
        {
            if (path == null) {
                if (schemeSpecificPart == null) {
                    if (other.schemeSpecificPart == null) {
                        return Objects.equals(query, other.query)
                            && authorityEquals(other);
                    } else {
                        return sspEquals(other);
                    }
                } else if (other.schemeSpecificPart == null) {
                    return sspEquals(other);
                } else {
                    return schemeSpecificPart.equals(other.schemeSpecificPart);
                }
            } else {
                return path.equals(other.path)
                    && Objects.equals(query, other.query)
                    && authorityEquals(other);
            }
        } else {
            return false;
        }
    }

    @Override
    public boolean equals(final Object o) {
        return o instanceof FastUri && equalsInternal((FastUri) o);
    }

    public String describe() {
        StringBuilder sb = new StringBuilder();
        describe(sb);
        return new String(sb);
    }

    public void describe(final StringBuilder sb) {
        sb.append('(');
        if (scheme != null) {
            sb.append("scheme=");
            sb.append(scheme);
            sb.append(',');
        }
        // ssp can't be null, unless set manually or URI is normalized, in
        // which case path is not null
        sb.append("ssp=");
        appendSchemeSpecificPart(sb, false);
        if (userInfo != null) {
            sb.append(",user=");
            sb.append(userInfo);
        }
        if (host != null) {
            sb.append(",host=");
            sb.append(host);
        }
        if (port != -1) {
            sb.append(",port=");
            sb.append(port);
        }
        if (authority != null) {
            sb.append(",authority=");
            sb.append(authority);
        }
        if (path != null) {
            sb.append(",path=");
            sb.append(path);
        }
        if (query != null) {
            sb.append(",query=");
            sb.append(query);
        }
        if (fragment != null) {
            sb.append(",fragment=");
            sb.append(fragment);
        }
        sb.append(')');
    }

    private static int needsNormalization(final String path) {
        int len = path.length();
        int start = 0;
        // Failfast: find any '/.', './' or '//'
        while (true) {
            int slash = path.indexOf('/', start);
            if (slash == -1) {
                return -1;
            }
            if (slash > 0) {
                if (path.charAt(slash - 1) == '.') {
                    break;
                }
            }
            int next = slash + 1;
            if (next == len) {
                return -1;
            }
            char nextChar = path.charAt(next);
            if (nextChar == '/' || nextChar == '.') {
                break;
            }
            start = next;
        }

        boolean normalized = true;
        int segmentsCount = 0;
        int pos = 0;
        int end = len - 1;

        // Skip initial slashes
        while (pos <= end) {
            if (path.charAt(pos) != '/') {
                break;
            }
            ++pos;
        }

        if (pos > 1) {
            normalized = false;
        }

        while (pos <= end) {
            // Looking for '.' or '..'
            if (path.charAt(pos) == '.') {
                if (pos == end) {
                    normalized = false;
                } else {
                    int next = pos + 1;
                    char nextChar = path.charAt(next);
                    if (nextChar == '/'
                        || (nextChar == '.'
                            && (next == end || path.charAt(next + 1) == '/')))
                    {
                        normalized = false;
                    }
                }
            }
            ++segmentsCount;
            // Find beginning of the next segment
            while (pos <= end) {
                if (path.charAt(pos++) == '/') {
                    // Skip redundant slashes
                    while (pos <= end && path.charAt(pos) == '/') {
                        normalized = false;
                        ++pos;
                    }
                    break;
                }
            }
        }
        if (normalized) {
            return -1;
        } else {
            return segmentsCount;
        }
    }

    private static void splitSegments(
        final char[] path,
        final int[] segments)
    {
        int pos = 0;
        int end = path.length - 1;

        // Skip initial slashes
        while (pos <= end && path[pos] == '/') {
            path[pos++] = '\0';
        }

        int segmentsPos = 0;
        while (pos <= end) {
            // Store start of segment
            segments[segmentsPos++] = pos++;
            // Find next segment
            while (pos <= end) {
                if (path[pos++] == '/') {
                    path[pos - 1] = '\0';
                    // Skip redundant slashes
                    while (pos <= end && path[pos] == '/') {
                        path[pos++] = '/';
                    }
                    break;
                }
            }
        }
    }

    private static void removeDots(final char[] path, final int[] segments) {
        int segmentsCount = segments.length;
        int end = path.length - 1;
        for (int i = 0; i < segmentsCount; ++i) {
            // Number of dots in current segment: 0, 1 or 2
            int dots = 0;

            // Find next '.' or '..'
            do {
                int pos = segments[i];
                if (path[pos] == '.') {
                    if (pos == end) {
                        dots = 1;
                        break;
                    } else {
                        int next = pos + 1;
                        char nextChar = path[next];
                        if (nextChar == '\0') {
                            dots = 1;
                            break;
                        } else if (nextChar == '.'
                            && (next == end || path[next + 1] == '\0'))
                        {
                            dots = 2;
                            break;
                        }
                    }
                }
                ++i;
            } while (i < segmentsCount);

            // No more removable segments left
            if (i > segmentsCount || dots == 0) {
                break;
            }

            if (dots == 1) {
                // Remove this '.'
                segments[i] = -1;
            } else { // dots == 2
                // Find preceding segment
                for (int j = i - 1; j >=0; --j) {
                    int pos = segments[j];
                    if (pos != -1) {
                        // Check that this is non-.. segment
                        if (path[pos] != '.'
                            || path[pos + 1] != '.'
                            || path[pos + 2] != '\0')
                        {
                            // And remove it
                            segments[i] = -1;
                            segments[j] = -1;
                        }
                        break;
                    }
                }
            }
        }
    }

    private static int joinSegments(
        final char[] path,
        final int[] segments)
    {
        int pos;
        if (path[0] == '\0') {
            // Restore initial slash
            path[0] = '/';
            pos = 1;
        } else {
            pos = 0;
        }
        int end = path.length - 1;
        int segmentsCount = segments.length;
        for (int i = 0; i < segmentsCount; ++i) {
            int segmentPos = segments[i];
            if (segmentPos != -1) {
                if (pos == segmentPos) {
                    // Segment is at right place, skip it
                    while (pos <= end && path[pos] != '\0') {
                        ++pos;
                    }
                    if (pos <= end) {
                        // Restore final slash
                        path[pos++] = '/';
                    }
                } else {
                    // Shiftcopy segment
                    while (segmentPos <= end && path[segmentPos] != '\0') {
                        path[pos++] = path[segmentPos++];
                    }
                    if (segmentPos <= end) {
                        // Restore final slash
                        path[pos++] = '/';
                    }
                }
            }
        }
        return pos;
    }

    public static String normalizePath(String path) {
        int segmentsCount = needsNormalization(path);
        if (segmentsCount == -1) {
            return path;
        }
        char[] buf = path.toCharArray();
        int[] segments = new int[segmentsCount];
        splitSegments(buf, segments);
        removeDots(buf, segments);
        String normalized = new String(buf, 0, joinSegments(buf, segments));
        if (normalized.equals(path)) {
            normalized = path;
        }
        return normalized;
    }

    @SuppressWarnings("ReferenceEquality")
    public FastUri normalize() {
        FastUri result;
        if (path == null || path.isEmpty()) {
            result = this;
        } else {
            String normalizedPath = normalizePath(path);
            if (normalizedPath == path) {
                result = this;
            } else {
                result = new FastUri(
                    scheme,
                    userInfo,
                    host,
                    port,
                    normalizedPath,
                    query,
                    fragment,
                    null,
                    authority);
            }
        }
        return result;
    }

    // RFC2396 §5.2
    // DEVIATION: base uri expected to be normalized
    public FastUri resolve(final FastUri child) {
        String childPath = child.path;
        if (path == null || childPath == null || child.scheme != null) {
            return child;
        }

        if (child.authority != null) {
            return new FastUri(
                scheme,
                child.userInfo,
                child.host,
                child.port,
                childPath,
                child.query,
                child.fragment,
                null,
                child.authority);
        }

        int childPathLength = childPath.length();

        // Check that this is a lone fragment
        // Scheme and authority is already null, paths already non-null
        if (child.fragment != null
            && child.query == null
            && childPathLength == 0)
        {
            if (child.fragment.equals(fragment)) {
                return this;
            }
            return new FastUri(
                scheme,
                userInfo,
                host,
                port,
                path,
                query,
                child.fragment,
                null,
                authority);
        }

        // RFC2396 §5.2 (6)
        String resolvedPath;
        if (childPathLength == 0) {
            resolvedPath = path.substring(0, path.lastIndexOf('/') + 1);
        } else if (childPath.charAt(0) == '/') {
            // Child path is absolute
            resolvedPath = normalizePath(childPath);
        } else {
            int slash = path.lastIndexOf('/');
            if (slash == -1) {
                resolvedPath = childPath;
            } else {
                ++slash;
                StringBuilder sb = new StringBuilder(slash + childPathLength);
                sb.append(path, 0, slash);
                sb.append(childPath);
                resolvedPath = new String(sb);
            }
            resolvedPath = normalizePath(resolvedPath);
        }

        return new FastUri(
            scheme,
            userInfo,
            host,
            port,
            resolvedPath,
            child.query,
            child.fragment,
            null,
            authority);
    }
}

