package ru.yandex.net.uri.fast;

import java.net.IDN;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Locale;

import ru.yandex.detect.locale.LocaleDetector;
import ru.yandex.util.match.tables.MatchTables;
import ru.yandex.util.string.HexStrings;
import ru.yandex.util.string.StringUtils;

public class FastUriParser {
    private static final int PCT_LEN = 3;
    private static final int MIN_NON_CONTROL = 161;
    private static final int MAX_BYTE = 255;
    private static final int IPV6_LENGTH = 16;

    private static final char[] HEX_CHARS = HexStrings.UPPER.chars();
    private static final int SHIFT = HexStrings.SHIFT;
    private static final int MASK = HexStrings.MASK;

    private static final boolean[] PCT = MatchTables.createMatchTable("%");
    private static final boolean[] DIGIT =
        MatchTables.createMatchTable("1234567890");
    private static final boolean[] HEX_DIGIT =
        MatchTables.createMatchTable("1234567890ABCDEFabcdef");
    private static final boolean[] OCT_DIGIT =
        MatchTables.createMatchTable("12345670");
    private static final boolean[] PATH_QUERY_FRAGMENT =
        MatchTables.createMatchTable("/?#");
    private static final boolean[] ALPHA = MatchTables.createMatchTable(
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz");
    private static final boolean[] ALPHANUM =
        MatchTables.mergeMatchTables(DIGIT, ALPHA);
    private static final boolean[] SCHEME =
        MatchTables.mergeMatchTables(
            ALPHA,
            DIGIT,
            MatchTables.createMatchTable("+-."));
    private static final boolean[] MARK =
        MatchTables.createMatchTable("-_.!~*'()");
    private static final boolean[] RESERVED =
        MatchTables.createMatchTable("$&+,/:;=?@[]");
    private static final boolean[] UNRESERVED =
        MatchTables.mergeMatchTables(DIGIT, ALPHA, MARK);
    private static final boolean[] URIC =
        MatchTables.mergeMatchTables(RESERVED, UNRESERVED);
    private static final boolean[] URIC_AND_PCT =
        MatchTables.mergeMatchTables(URIC, PCT);
    private static final boolean[] PATH =
        MatchTables.mergeMatchTables(
            UNRESERVED,
            MatchTables.createMatchTable("$&+,/:;=@"));
    private static final boolean[] PATH_AND_PCT =
        MatchTables.mergeMatchTables(PATH, PCT);
    private static final boolean[] DASH = MatchTables.createMatchTable("-");
    private static final boolean[] DOT = MatchTables.createMatchTable(".");
    private static final boolean[] USERINFO =
        MatchTables.mergeMatchTables(UNRESERVED,
                MatchTables.createMatchTable("$&+,:;="));
    private static final boolean[] REG_NAME =
        MatchTables.mergeMatchTables(
            UNRESERVED,
            MatchTables.createMatchTable("$&+,:;=@"));
    private static final boolean[] SERVER =
        MatchTables.mergeMatchTables(
            USERINFO,
            ALPHA,
            DIGIT,
            DASH,
            MatchTables.createMatchTable(".:@[]"));
    private static final boolean[] SERVER_PCT =
        MatchTables.mergeMatchTables(SERVER, PCT);
    private static final boolean[] DIGIT_OR_DOT =
        MatchTables.mergeMatchTables(DIGIT, DOT);
    private static final boolean[] HEX_OCTETS =
        MatchTables.mergeMatchTables(
            HEX_DIGIT,
            DOT,
            MatchTables.createMatchTable("Xx"));

    private static final String IPV4_ADDRESS = "IPv4 address";
    private static final String MALFORMED_IPV4 = "Malformed IPv4 address";
    private static final String MALFORMED_IPV6 = "Malformed IPv6 address";
    private static final String HEX_DIGITS_OR_IPV4 =
        "hex digits or IPv4 address";
    private static final String ILLEGAL_CHAR_IN_HOSTNAME =
        "Illegal character in hostname";
    private static final String EXPECTED_SCHEME = "Expected scheme name";
    private static final String ILLEGAL_CHAR = "Illegal character in ";
    private static final String HEXADECIMAN_TOO_LONG =
        "IPv6 hexadecimal digit sequence too long";

    private static final char[] EMPTY_BUF = new char[0];

    private final String uri;
    private final char[] input;
    private final int off;
    private final int len;
    private String scheme = null;
    private String userInfo = null;
    private String host = null;
    private String path = null;
    private String query = null;
    private String fragment = null;
    private String schemeSpecificPart = null;
    private String authority = null;
    private int port = -1;
    private int ipv6ByteCount;
    private StringBuilder sb = null;
    private char[] buf = EMPTY_BUF;

    public FastUriParser(final String uri) {
        this.uri = uri;
        input = uri.toCharArray();
        off = 0;
        len = input.length;
    }

    public FastUriParser(final char[] input) {
        this(input, 0, input.length);
    }

    public FastUriParser(final char[] input, final int off, final int len) {
        this.input = input;
        this.off = off;
        this.len = len;
        uri = null;
    }

    public FastUriParser(final StringBuilder sb) {
        uri = null;
        off = 0;
        len = sb.length();
        input = new char[len];
        sb.getChars(0, len, input, 0);
    }

    private String uri() {
        if (uri == null) {
            return new String(input, off, len);
        } else {
            return uri;
        }
    }

    private void prepareSb(final int capacity) {
        if (sb == null) {
            sb = new StringBuilder(capacity);
        } else {
            if ((sb.capacity() << 1) < capacity) {
                sb.ensureCapacity(capacity);
            }
            sb.setLength(0);
        }
    }

    private void prepareBuf(final int capacity) {
        if (buf.length < capacity) {
            buf = new char[capacity];
        }
    }

    private boolean isAt(final int start, final int end, final char c) {
        return start < end && input[start] == c;
    }

    private String substring(final int start, final int end) {
        return new String(input, start, end - start);
    }

    private String substring(
        final int start,
        final int end,
        final boolean[] table)
    {
        for (int i = start; i < end; ++i) {
            char c = input[i];
            if (!MatchTables.match(table, c)) {
                // Whoops, encoding required
                int prefixLen = i - start;
                if (c >= 0x80) {
                    prepareSb(prefixLen + (end - i) * 6);
                    sb.append(input, start, prefixLen);
                    return substringEncodedUTF(i, end, table);
                } else {
                    prepareSb(prefixLen + (end - i) * 3);
                    sb.append(input, start, prefixLen);
                    return substringEncodedASCII(i, end, table);
                }
            }
        }
        return substring(start, end);
    }

    private String substringEncodedASCII(
        final int start,
        final int end,
        final boolean[] table)
    {
        for (int i = start; i < end; ++i) {
            char c = input[i];
            if (MatchTables.match(table, c)) {
                sb.append(c);
            } else {
                if (c >= 0x80) {
                    return substringEncodedUTF(i, end, table);
                } else {
                    sb.append('%');
                    sb.append(HEX_CHARS[c >> SHIFT]);
                    sb.append(HEX_CHARS[c & MASK]);
                }
            }
        }
        return new String(sb);
    }

    private String substringEncodedUTF(
        final int start,
        final int end,
        final boolean[] table)
    {
        byte[] bytes = StringUtils.getUtf8Bytes(substring(start, end));
        for (int i = 0; i < bytes.length; ++i) {
            int b = bytes[i] & 0xff;
            if (b < table.length && table[b]) {
                sb.append((char) b);
            } else {
                sb.append('%');
                sb.append(HEX_CHARS[b >> SHIFT]);
                sb.append(HEX_CHARS[b & MASK]);
            }
        }
        return new String(sb);
    }

    // CSOFF: ParameterNumber
    private int scan(final int start, final int end, final char c) {
        if (start < end && input[start] == c) {
            return start + 1;
        } else {
            return start;
        }
    }
    // CSON: ParameterNumber

    private int scan(
        final int start,
        final int end,
        final boolean[] table)
    {
        int p = start;
        for (; p < end; ++p) {
            if (!MatchTables.match(table, input[p])) {
                break;
            }
        }
        return p;
    }

    // CSOFF: ParameterNumber
    private int scan(
        final int start,
        final int end,
        final boolean[] err,
        final char stop)
    {
        int p = start;
        while (p < end) {
            char c = input[p];
            if (MatchTables.match(err, c)) {
                return -1;
            }
            if (stop == c) {
                break;
            }
            ++p;
        }
        return p;
    }
    // CSON: ParameterNumber

    private int scanStop(
        final int start,
        final int end,
        final char stop)
    {
        int p = start;
        while (p < end) {
            if (input[p] == stop) {
                break;
            }
            ++p;
        }
        return p;
    }

    // CSOFF: ParameterNumber
    private int scanStop(
        final int start,
        final int end,
        final char stop1,
        final char stop2)
    {
        int p = start;
        while (p < end) {
            char c = input[p];
            if (c == stop1 || c == stop2) {
                break;
            }
            ++p;
        }
        return p;
    }
    // CSON: ParameterNumber

    private int scanStop(
        final int start,
        final int end,
        final boolean[] stop)
    {
        int p = start;
        while (p < end) {
            if (MatchTables.match(stop, input[p])) {
                break;
            }
            ++p;
        }
        return p;
    }

    private int scanEscaped(
        final int start,
        final int end,
        final char first)
        throws URISyntaxException
    {
        int result;
        if (first == '%') {
            int max = start + PCT_LEN;
            if (max <= end
                && MatchTables.match(HEX_DIGIT, input[start + 1])
                && MatchTables.match(HEX_DIGIT, input[start + 2]))
            {
                result = max;
            } else {
                throw new URISyntaxException(
                    uri(),
                    "Malformed escape char",
                    start);
            }
        } else if (first >= MIN_NON_CONTROL
            && !Character.isSpaceChar(first)
            && !Character.isISOControl(first))
        {
            result = start + 1;
        } else {
            result = start;
        }
        return result;
    }

    private int scanEscaped(
        final int start,
        final int end,
        final boolean[] table)
        throws URISyntaxException
    {
        int p = start;
        while (p < end) {
            char c = input[p];
            if (MatchTables.match(table, c)) {
                ++p;
            } else {
                int q = scanEscaped(p, end, c);
                if (q > p) {
                    p = q;
                } else {
                    break;
                }
            }
        }
        return p;
    }

    // CSOFF: ParameterNumber
    private void checkChar(
        final int start,
        final int end,
        final char c,
        final String what)
        throws URISyntaxException
    {
        if (start >= end || input[start] != c) {
            throw new URISyntaxException(
                uri(),
                ILLEGAL_CHAR + what,
                start);
        }
    }

    private void checkChars(
        final int start,
        final int end,
        final boolean[] table,
        final String what)
        throws URISyntaxException
    {
        int p = scan(start, end, table);
        if (p < end) {
            throw new URISyntaxException(
                uri(),
                ILLEGAL_CHAR + what,
                p);
        }
    }

    private void checkEscapedChars(
        final int start,
        final int end,
        final boolean[] table,
        final String what)
        throws URISyntaxException
    {
        int p = scanEscaped(start, end, table);
        if (p < end) {
            throw new URISyntaxException(
                uri(),
                ILLEGAL_CHAR + what,
                p);
        }
    }
    // CSON: ParameterNumber

    public FastUri parse() throws URISyntaxException {
        int ssp;
        int end = off + len;
        int p = scan(off, end, PATH_QUERY_FRAGMENT, ':');
        if (p >= off && isAt(p, end, ':')) {
            if (p == off) {
                throw new URISyntaxException(
                    uri(),
                    EXPECTED_SCHEME,
                    0);
            }
            if (!MatchTables.match(ALPHA, input[off])) {
                throw new URISyntaxException(
                    uri(),
                    EXPECTED_SCHEME,
                    off);
            }
            checkChars(off + 1, p, SCHEME, "scheme name");
            scheme = substring(off, p);
            ++p;
            ssp = p;
            if (isAt(p, end, '/')) {
                p = parseHierarchical(p, end);
            } else {
                int q = scanStop(p, end, '#');
                if (q <= p) {
                    throw new URISyntaxException(
                        uri(),
                        "Expected scheme specific part",
                        p);
                }
                // DEVIATION: Do not validate ssp, just encode everything
                // checkEscapedChars(p, q, URIC, "opaque part");
                p = q;
            }
        } else {
            ssp = off;
            p = parseHierarchical(off, end);
        }
        schemeSpecificPart = substring(ssp, p, URIC_AND_PCT);
        if (isAt(p, end, '#')) {
            // DEVIATION: Do not validate fragment, just encode everything
            // Also, use URIC instead of FRAGMENT so # will be encoded, too
            // java.net.URI doesn't allow two # in uri
            // int fragmentEnd = scanEscaped(next, end, FRAGMENT);
            int next = p + 1;
            fragment = substring(next, end, URIC_AND_PCT);
            p = end;
        }
        if (p < end) {
            throw new URISyntaxException(uri(), "end of URI", p);
        }
        return new FastUri(
            scheme,
            userInfo,
            host,
            port,
            path,
            query,
            fragment,
            schemeSpecificPart,
            authority);
    }

    private int parseHierarchical(final int start, final int end)
        throws URISyntaxException
    {
        int p = start;
        if (isAt(p, end, '/') && isAt(p + 1, end, '/')) {
            p += 2;
            int q = scanStop(p, end, PATH_QUERY_FRAGMENT);
            if (q > p) {
                p = parseAuthority(p, q);
            } else if (q < end) {
                // DEVIATION: Allow empty authority prior to non-empty
                // path, query component or fragment identifier
                authority = "";
            } else {
                throw new URISyntaxException(
                    uri(),
                    "Expecting authority",
                     p);
            }
        }
        int q = scanStop(p, end, '?', '#');
        // DEVIATION: Do not validate PATH, just encode everyting
        // checkEscapedChars(p, q, PATH, "path");
        path = substring(p, q, PATH_AND_PCT);
        p = q;
        if (isAt(p, end, '?')) {
            ++p;
            q = scanStop(p, end, '#');
            // DEVIATION: Do not validate QUERY, just encode everyting
            // checkEscapedChars(p, q, URIC, "query");
            query = substring(p, q, URIC_AND_PCT);
            p = q;
        }
        return p;
    }

    private int parseAuthority(final int start, final int end)
        throws URISyntaxException
    {
        boolean serverChars;
        if (scanStop(start, end, ']') > start) {
            serverChars = scanEscaped(start, end, SERVER_PCT) == end;
        } else {
            serverChars = scanEscaped(start, end, SERVER) == end;
        }

        boolean regChars = scanEscaped(start, end, REG_NAME) == end;
        if (regChars && !serverChars) {
            authority = substring(start, end);
            return end;
        }

        int q;
        if (serverChars) {
            q = parseServer(start, end);
            if (q < end) {
                throw new URISyntaxException(
                    uri(),
                    "Expected end of authority",
                    q);
            }
            authority = substring(start, end);
        } else {
            q = start;
        }

        if (q < end) {
            if (regChars) {
                authority = substring(start, end);
            } else {
                throw new URISyntaxException(
                    uri(),
                    "Illegal character in authority",
                    q);
            }
        }

        return end;
    }

    private int parseServer(final int start, final int end)
        throws URISyntaxException
    {
        int p = start;
        int q = scan(p, end, PATH_QUERY_FRAGMENT, '@');
        if (q >= p && isAt(q, end, '@')) {
            checkEscapedChars(p, q, USERINFO, "user info");
            userInfo = substring(p, q);
            p = q + 1;
        }

        if (isAt(p, end, '[')) {
            ++p;
            q = scan(p, end, PATH_QUERY_FRAGMENT, ']');
            if (q > p && isAt(q, end, ']')) {
                int r = scanStop(p, q, '%');
                if (r > p) {
                    // XXX
                    parseIPv6Reference(p, r);
                    int scope = r + 1;
                    if (scope == q) {
                        throw new URISyntaxException(
                            uri(),
                            "scope id expected",
                            r);
                    }
                    checkChars(scope, q, ALPHANUM, "scope id");
                } else {
                    parseIPv6Reference(p, q);
                }
                int addrEnd = q + 1;
                host = substring(p - 1, addrEnd);
                p = addrEnd;
            } else {
                throw new URISyntaxException(
                    uri(),
                    "Expecting closing bracket for IPv6 address",
                    q);
            }
        } else {
            q = parseIPv4Address(p, end);
            if (q <= p) {
                q = parseHostname(p, end);
            }
            p = q;
        }

        if (isAt(p, end, ':')) {
            ++p;
            q = scanStop(p, end, '/');
            if (q > p) {
                checkChars(p, q, DIGIT, "port number");
                try {
                    port = Integer.parseInt(substring(p, q));
                } catch (NumberFormatException e) {
                    throw new URISyntaxException(
                        uri(),
                        "Malformed port number",
                        p);
                }
                p = q;
            }
        }
        if (p < end) {
            throw new URISyntaxException(
                uri(),
                "Expecting port number",
                p);
        }
        return p;
    }

    private int scanOctet(final int start, final int end) {
        if (start < end) {
            int m;
            int value;
            if (input[start] == '0') {
                int next = start + 1;
                if (next < end) {
                    char c = input[next];
                    if (c == 'x' || c == 'X') {
                        int hexStart = next + 1;
                        m = scan(hexStart, end, HEX_DIGIT);
                        value = Integer.parseInt(substring(hexStart, m), 16);
                    } else {
                        m = scan(next, end, OCT_DIGIT);
                        if (m == next) {
                            value = 0;
                        } else {
                            value = Integer.parseInt(substring(next, m), 8);
                        }
                    }
                } else {
                    m = next;
                    value = 0;
                }
            } else {
                m = scan(start, end, DIGIT);
                value = Integer.parseInt(substring(start, m));
            }
            if (value <= MAX_BYTE) {
                return m;
            }
        }
        return -1;
    }

    private int scanIPv4Address(final int start, final int end)
        throws URISyntaxException
    {
        int m = scan(start, end, HEX_OCTETS);
        if (m <= start) {
            return -1;
        }
        if (scanStop(start, m, '.') == m) {
            // there is no dots, so this is a single number ipv4
            // like 0x97f87316, 022776071426 or 2549642006,
            try {
                if (input[start] == '0') {
                    int next = start + 1;
                    if (next < m) {
                        char c = input[next];
                        if (c == 'x' || c == 'X') {
                            // hex form
                            Integer.parseUnsignedInt(
                                substring(start + 2, m),
                                16);
                        } else {
                            // octal form
                            Integer.parseUnsignedInt(substring(next, m), 8);
                        }
                    } // else it is just zero
                } else {
                    Integer.parseUnsignedInt(substring(start, m));
                }
                // Successfully parsed
                return m;
            } catch (NumberFormatException e) {
            }
            // but also this can be a plain hostname like 0xff0xdeadbeef
            return -1;
        } else {
            // try to parse as octets
            try {
                int p = start;
                int q = scanOctet(p, m);
                if (q == -1) {
                    return -1;
                }
                for (int i = 0; i <= 2; ++i) {
                    p = q;
                    if (p < m && input[p] == '.') {
                        q = scanOctet(p + 1, m);
                        if (q == -1) {
                            return -1;
                        }
                    } else {
                        break;
                    }
                }
                if (q == m) {
                    return m;
                }
            } catch (NumberFormatException e) {
                // parse failed, this could be host like 0xdead.0xbeef
            }
            return -1;
        }
    }

    private int scanByte(final int start, final int end)
        throws URISyntaxException
    {
        int q = scan(start, end, DIGIT);
        if (q > start) {
            try {
                int b = Integer.parseInt(substring(start, q));
                if (b <= MAX_BYTE) {
                    return q;
                }
            } catch (NumberFormatException e) {
            }
        }
        throw new URISyntaxException(
            uri(),
            "Malformed byte value",
            start);
    }

    private int scanIPv4AddressStrict(final int start, final int end)
        throws URISyntaxException
    {
        if (scan(start, end, DIGIT_OR_DOT) != end) {
            return -1;
        }
        int p = start;
        int q = scanByte(p, end);
        for (int i = 0; i <= 2; ++i) {
            p = q;
            checkChar(p, end, '.', IPV4_ADDRESS);
            q = scanByte(p + 1, end);
        }
        if (q < end) {
            throw new URISyntaxException(
                uri(),
                MALFORMED_IPV4,
                q);
        }
        return end;
    }

    private int takeIPv4Address(
        final int start,
        final int end,
        final String expected)
        throws URISyntaxException
    {
        int p = scanIPv4AddressStrict(start, end);
        if (p <= start) {
            throw new URISyntaxException(
                uri(),
                "Expecting " + expected,
                start);
        }
        return p;
    }

    private int parseIPv4Address(final int start, final int end)
        throws URISyntaxException
    {
        int p;
        try {
            p = scanIPv4Address(start, end);
        } catch (URISyntaxException e) {
            return -1;
        }
        if (p > start && p < end && input[p] != ':') {
            p = -1;
        }
        if (p > start) {
            host = substring(start, p);
        }
        return p;
    }

    private int parseHostname(final int start, final int end)
        throws URISyntaxException
    {
        int p = start;
        int last = -1;
        do {
            int q = p;
            for (; q < end; ++q) {
                char c = input[q];
                if (!Character.isLetterOrDigit(c) && c != '_') {
                    break;
                }
            }
            if (q == p) {
                break;
            }
            last = p;
            p = q;
            for (; q < end; ++q) {
                char c = input[q];
                if (!Character.isLetterOrDigit(c) && c != '-' && c != '_') {
                    break;
                }
            }
            if (q > p) {
                char lastChar = input[q - 1];
                if (lastChar == '-') {
                    throw new URISyntaxException(
                        uri(),
                        ILLEGAL_CHAR_IN_HOSTNAME,
                        q - 1);
                }
                p = q;
            }
            q = scan(p, end, '.');
            if (q == p) {
                break;
            }
            p = q;
        } while (p < end);

        if (p < end && input[p] != ':') {
            throw new URISyntaxException(
                uri(),
                ILLEGAL_CHAR_IN_HOSTNAME,
                p);
        }
        if (last < 0) {
            throw new URISyntaxException(
                uri(),
                "Expecting hostname",
                start);
        }
        if (last > start && !Character.isLetter(input[last])) {
            throw new URISyntaxException(
                uri(),
                ILLEGAL_CHAR_IN_HOSTNAME,
                last);
        }
        try {
            String lowercaseHost = lowercaseHost(start, p);
            host = IDN.toASCII(lowercaseHost, IDN.ALLOW_UNASSIGNED);
            if (!lowercaseHost.equals(host)) {
                String unicodeHost = IDN.toUnicode(host, IDN.ALLOW_UNASSIGNED);
                if (!unicodeHost.equals(lowercaseHost)) {
                    throw new URISyntaxException(
                        uri(),
                        "toUnicode(toASCII(\"" + lowercaseHost
                        + "\")) returned \"" + unicodeHost + '"',
                        start);
                }
            }
        } catch (RuntimeException e) {
            URISyntaxException ex = new URISyntaxException(
                uri(),
                "Invalid hostname <" + substring(start, p) + '>',
                start);
            ex.initCause(e);
            throw ex;
        }
        return p;
    }

    private int scanHexSeq(final int start, final int end)
        throws URISyntaxException
    {
        int q = scan(start, end, HEX_DIGIT);
        if (q <= start || isAt(q, end, '.')) {
            return -1;
        }
        if (q > start + 2 + 2) {
            throw new URISyntaxException(
                uri(),
                HEXADECIMAN_TOO_LONG,
                start);
        }
        ipv6ByteCount += 2;
        int p = q;
        while (p < end) {
            if (!isAt(p, end, ':') || isAt(p + 1, end, ':')) {
                // ::
                break;
            }
            ++p;
            q = scan(p, end, HEX_DIGIT);
            if (q <= p) {
                throw new URISyntaxException(
                    uri(),
                    "Expecting digits for IPv6 address",
                    p);
            }
            if (isAt(q, end, '.')) {
                --p;
                break;
            }
            if (q > p + 2 + 2) {
                throw new URISyntaxException(
                    uri(),
                    HEXADECIMAN_TOO_LONG,
                    p);
            }
            ipv6ByteCount += 2;
            p = q;
        }
        return p;
    }

    private int scanHexPost(final int start, final int end)
        throws URISyntaxException
    {
        if (start == end) {
            return start;
        }
        int q = scanHexSeq(start, end);
        int p;
        if (q > start) {
            p = q;
            if (isAt(p, end, ':')) {
                p = takeIPv4Address(p + 1, end, HEX_DIGITS_OR_IPV4);
                ipv6ByteCount += 2 + 2;
            }
        } else {
            p = takeIPv4Address(start, end, HEX_DIGITS_OR_IPV4);
            ipv6ByteCount += 2 + 2;
        }
        return p;
    }

    private int parseIPv6Reference(final int start, final int end)
        throws URISyntaxException
    {
        ipv6ByteCount = 0;
        boolean compressedZeroes = false;
        int q = scanHexSeq(start, end);
        int p;
        if (q > start) {
            p = q;
            if (isAt(p, end, ':')) {
                int next = p + 1;
                if (isAt(next, end, ':')) {
                    compressedZeroes = true;
                    p = scanHexPost(p + 2, end);
                } else {
                    // XXX?
                    p = takeIPv4Address(next, end, IPV4_ADDRESS);
                    ipv6ByteCount += 2 + 2;
                }
            }
        } else {
            if (isAt(start, end, ':') && isAt(start + 1, end, ':')) {
                compressedZeroes = true;
                p = scanHexPost(start + 2, end);
            } else {
                p = start;
            }
        }

        if (p < end) {
            throw new URISyntaxException(uri(), MALFORMED_IPV6, start);
        }
        if (ipv6ByteCount > IPV6_LENGTH) {
            throw new URISyntaxException(
                uri(),
                "IPv6 address too long",
                start);
        }
        if (!compressedZeroes && ipv6ByteCount < IPV6_LENGTH) {
            throw new URISyntaxException(
                uri(),
                "IPv6 address too short",
                start);
        }
        if (compressedZeroes && ipv6ByteCount == IPV6_LENGTH) {
            throw new URISyntaxException(uri(), MALFORMED_IPV6, start);
        }
        return p;
    }

    private void lowercaseHostPart(
        final int[] localeChars,
        final int off,
        final int len)
    {
        char[] newChars;
        int newOff;
        int newLen;
        if (len >= 4
            && (input[off] == 'x' || input[off] == 'X')
            && (input[off + 1] == 'n' || input[off + 1] == 'N')
            && input[off + 2] == '-'
            && input[off + 3] == '-')
        {
            String str = IDN.toUnicode(
                new String(input, off, len),
                IDN.ALLOW_UNASSIGNED);
            int strLen = str.length();
            prepareBuf(strLen);
            str.getChars(0, strLen, buf, 0);
            newChars = buf;
            newOff = 0;
            newLen = strLen;
        } else {
            newChars = input;
            newOff = off;
            newLen = len;
        }
        boolean allLower = true;
        for (int i = 0; i < newLen && allLower; ++i) {
            char c = newChars[i + newOff];
            if (Character.isUpperCase(c)) {
                allLower = false;
            }
        }
        if (allLower) {
            sb.append(newChars, newOff, newLen);
        } else {
            Arrays.fill(localeChars, 0);
            Locale locale = LocaleDetector.INSTANCE.detectLocale(
                localeChars,
                newChars,
                newOff,
                newLen);
            if (locale == null) {
                locale = Locale.ROOT;
            }
            String str = new String(newChars, newOff, newLen);
            sb.append(str.toLowerCase(locale));
        }
    }

    private String lowercaseHost(final int start, final int end) {
        prepareSb(end - start);
        int[] localeChars = LocaleDetector.INSTANCE.prepareLocaleChars();
        int prev = start;
        for (int i = start; i < end; ++i) {
            char c = input[i];
            if (c == '.') {
                lowercaseHostPart(localeChars, prev, i - prev);
                sb.append('.');
                prev = i + 1;
            }
        }
        if (prev == start) {
            // This hostname has no delimiter, most probably this is some
            // custom protocol, so not lowercasing should be performed
            return substring(start, end);
        } else {
            lowercaseHostPart(localeChars, prev, end - prev);
            return new String(sb);
        }
    }
}

