package ru.yandex.function;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.errorprone.annotations.NoAllocation;

public interface StringBuilderable {
    String NULL = "null";
    int DEFAULT_STRING_LENGTH = 32;
    int NULL_LENGTH = NULL.length();
    int MAX_SHORT_LENGTH = 6;
    int MAX_INT_LENGTH = 11;
    int MAX_LONG_LENGTH = 20;
    int MAX_DOUBLE_LENGTH = 26;

    @NoAllocation
    default int expectedStringLength() {
        return DEFAULT_STRING_LENGTH;
    }

    @Nonnull
    default String toStringFast() {
        StringBuilder sb = new StringBuilder(expectedStringLength());
        toStringBuilder(sb);
        return new String(sb);
    }

    void toStringBuilder(@Nonnull StringBuilder sb);

    static void toStringBuilder(
        @Nonnull final StringBuilder sb,
        @Nullable final StringBuilderable value)
    {
        if (value == null) {
            sb.append(NULL);
        } else {
            value.toStringBuilder(sb);
        }
    }

    static void toStringBuilder(
        @Nonnull final StringBuilder sb,
        @Nullable final List<? extends StringBuilderable> values)
    {
        if (values == null) {
            sb.append(NULL);
        } else {
            int size = values.size();
            sb.append('[');
            for (int i = 0; i < size; ++i) {
                if (i != 0) {
                    sb.append(',');
                }
                values.get(i).toStringBuilder(sb);
            }
            sb.append(']');
        }
    }

    static void toStringBuilder(
        @Nonnull final StringBuilder sb,
        @Nullable final Collection<? extends StringBuilderable> values)
    {
        if (values == null) {
            sb.append(NULL);
        } else {
            boolean empty = true;
            sb.append('[');
            for (StringBuilderable value: values) {
                if (empty) {
                    empty = false;
                } else {
                    sb.append(',');
                }
                value.toStringBuilder(sb);
            }
            sb.append(']');
        }
    }

    @NoAllocation
    static int calcExpectedStringLength(@Nullable final String str) {
        return calcExpectedStringLength(str, NULL_LENGTH);
    }

    @NoAllocation
    static int calcExpectedStringLength(
        @Nullable final String str,
        final int nullLength)
    {
        if (str == null) {
            return nullLength;
        } else {
            return str.length();
        }
    }

    @NoAllocation
    static int calcExpectedStringLength(@Nullable final Long value) {
        if (value == null) {
            return NULL_LENGTH;
        } else {
            return MAX_LONG_LENGTH;
        }
    }

    @NoAllocation
    static int calcExpectedStringLength(@Nullable final Double value) {
        if (value == null) {
            return NULL_LENGTH;
        } else {
            return MAX_DOUBLE_LENGTH;
        }
    }

    @NoAllocation
    static int calcExpectedStringLength(
        @Nullable final StringBuilderable value)
    {
        if (value == null) {
            return NULL_LENGTH;
        } else {
            return value.expectedStringLength();
        }
    }

    @NoAllocation
    static int calcExpectedStringLength(
        @Nullable final List<? extends StringBuilderable> values)
    {
        int len;
        if (values == null) {
            len = NULL_LENGTH;
        } else {
            int size = values.size();
            len = size + 1;
            for (int i = 0; i < size; ++i) {
                len += values.get(i).expectedStringLength();
            }
        }
        return len;
    }
}

