package ru.yandex.io;

import java.io.IOException;
import java.io.Writer;

// Writer implementation of .append(String) method may cause multiple
// allocations, this class reuses persistent buffer of limited size
public class AllocationlessWriter extends Writer {
    private static final char[] EMPTY_BUF = new char[0];
    private static final int DEFAULT_MAX_BUF_SIZE = 1024;

    private final Writer writer;
    private final int maxBufferSize;
    private char[] buf = EMPTY_BUF;

    public AllocationlessWriter(final Writer writer) {
        this(writer, DEFAULT_MAX_BUF_SIZE);
    }

    public AllocationlessWriter(final Writer writer, final int maxBufferSize) {
        this.writer = writer;
        this.maxBufferSize = maxBufferSize;
    }

    private void growBuf(final int len) {
        if (len > buf.length) {
            int newLen =
                Math.min(
                    maxBufferSize,
                    Math.max(buf.length << 1, len));
            if (newLen > buf.length) {
                buf = new char[newLen];
            }
        }
    }

    @Override
    public AllocationlessWriter append(final CharSequence csq)
        throws IOException
    {
        if (csq instanceof StringBuilder) {
            return append((StringBuilder) csq, 0, csq.length());
        } else {
            writer.append(csq);
            return this;
        }
    }

    @Override
    public AllocationlessWriter append(
        final CharSequence csq,
        final int start,
        final int end)
        throws IOException
    {
        if (csq instanceof StringBuilder) {
            return append((StringBuilder) csq, start, end);
        } else {
            writer.append(csq, start, end);
            return this;
        }
    }

    public AllocationlessWriter append(final StringBuilder sb)
        throws IOException
    {
        return append(sb, 0, sb.length());
    }

    public AllocationlessWriter append(
        final StringBuilder sb,
        final int start,
        final int end)
        throws IOException
    {
        growBuf(end - start);
        int pos = start;
        while (pos < end) {
            int len = Math.min(buf.length, end - pos);
            sb.getChars(pos, pos + len, buf, 0);
            writer.write(buf, 0, len);
            pos += len;
        }
        return this;
    }

    @Override
    public void write(final int ch) throws IOException {
        writer.write(ch);
    }

    @Override
    public void write(final char[] buf, final int off, final int len)
        throws IOException
    {
        writer.write(buf, off, len);
    }

    @Override
    public void write(
        final String str,
        final int off,
        final int len)
        throws IOException
    {
        growBuf(len);
        int pos = 0;
        while (pos < len) {
            int localLen = Math.min(buf.length, len - pos);
            str.getChars(off + pos, off + pos + localLen, buf, 0);
            writer.write(buf, 0, localLen);
            pos += localLen;
        }
    }

    @Override
    public void flush() throws IOException {
        writer.flush();
    }

    @Override
    public void close() throws IOException {
        writer.close();
    }
}
