package ru.yandex.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.PushbackInputStream;

public class BufferedPushbackInputStream extends PushbackInputStream {
    private static final int DEFAULT_BUFFER_SIZE = 2048;
    private static final int MIN_BUFFER_SIZE = 64;

    protected int end;

    public BufferedPushbackInputStream(final InputStream in) {
        this(in, DEFAULT_BUFFER_SIZE);
    }

    public BufferedPushbackInputStream(
        final InputStream in,
        final int size)
    {
        super(in, Math.max(size, MIN_BUFFER_SIZE));
        end = buf.length >> 2;
        pos = end;
    }

    private void ensureOpen() throws IOException {
        if (buf == null) {
            throw new IOException("Stream closed");
        }
    }

    // Preconditions:
    // 1. pos == end, i.e. no data in buffer available
    // 2. ensureOpen() is called prior to call to this function
    //
    // Postconditions:
    // 1. Buffer is filled with data from underlying stream.
    // 2. pos is set to buf.length >> 2
    private void fillBuffer() throws IOException {
        int newPos = buf.length >> 2;
        int capacity = buf.length - newPos;
        int read = in.read(buf, newPos, capacity);
        if (read > 0) {
            pos = newPos;
            end = pos + read;
        }
    }

    @Override
    public int available() throws IOException {
        return end - pos + in.available();
    }

    @Override
    @SuppressWarnings("UnsynchronizedOverridesSynchronized")
    public void close() throws IOException {
        buf = null;
        in.close();
    }

    @Override
    public void mark(final int markLimit) {
    }

    @Override
    public boolean markSupported() {
        return false;
    }

    @Override
    @SuppressWarnings("UnsynchronizedOverridesSynchronized")
    public void reset() throws IOException {
        throw new IOException("mark/reset not supported");
    }

    @Override
    public int read() throws IOException {
        ensureOpen();
        if (pos >= end) {
            fillBuffer();
            if (pos >= end) {
                return -1;
            }
        }
        return buf[pos++] & 0xff;
    }

    @Override
    public int read(final byte[] b) throws IOException {
        return read(b, 0, b.length);
    }

    @Override
    public int read(final byte[] b, final int off, final int len)
        throws IOException
    {
        ensureOpen();
        int available = end - pos;
        if (available <= 0) {
            if (len >= (buf.length >> 1)) {
                // We have no buffered data and read is big enough, so perform
                // direct read
                return in.read(b, off, len);
            }
            fillBuffer();
            available = end - pos;
            if (available <= 0) {
                return -1;
            }
        }
        int count = Math.min(available, len);
        System.arraycopy(buf, pos, b, off, count);
        pos += count;
        return count;
    }

    @Override
    public long skip(final long n) throws IOException {
        ensureOpen();
        int available = end - pos;
        if (available <= 0) {
            return in.skip(n);
        } else {
            long count = Math.min(available, n);
            pos += (int) count;
            return count;
        }
    }

    @Override
    public void unread(int b) throws IOException {
        ensureOpen();
        if (pos == 0) {
            int len = end - pos;
            byte[] newBuf = new byte[buf.length << 1];
            int newPos = buf.length >> 1;
            System.arraycopy(buf, pos, newBuf, newPos, len);
            pos = newPos;
            end = pos + len;
            buf = newBuf;
        }
        buf[--pos] = (byte) b;
    }

    @Override
    public void unread(byte[] b) throws IOException {
        unread(b, 0, b.length);
    }

    @Override
    public void unread(byte[] b, final int off, final int len)
        throws IOException
    {
        ensureOpen();
        if (pos < len) {
            int count = end - pos;
            int required = count + len;
            byte[] dstBuf;
            if (required <= buf.length) {
                dstBuf = buf;
            } else {
                dstBuf = new byte[Math.max(required, buf.length << 1)];
            }
            end = dstBuf.length;
            int newPos = end - count;
            System.arraycopy(buf, pos, dstBuf, newPos, count);
            pos = newPos;
            buf = dstBuf;
        }
        pos -= len;
        System.arraycopy(b, off, buf, pos, len);
    }
}

