package ru.yandex.io;

import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;

import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.ByteArrayVoidProcessor;
import ru.yandex.function.Processable;
import ru.yandex.function.Processor;
import ru.yandex.function.VoidProcessor;

public class DecodableByteArrayOutputStream
    extends OutputStream
    implements Processable<byte[]>, ByteArrayVoidProcessor<RuntimeException>
{
    private static final int DEFAULT_SIZE = 1024;
    private static final byte[] EMPTY_BUF = new byte[0];

    private int len = 0;
    private byte[] buf;

    public DecodableByteArrayOutputStream() {
        this(DEFAULT_SIZE);
    }

    public DecodableByteArrayOutputStream(final int size) {
        if (size == 0) {
            buf = EMPTY_BUF;
        } else {
            buf = new byte[size];
        }
    }

    public void ensureCapacity(final int size) {
        if (size > buf.length) {
            buf = Arrays.copyOf(buf, Math.max(size, len << 1));
        }
    }

    public void truncate(final int len, final byte delimiter) {
        if (len < this.len) {
            for (int i = len; i >= 0; --i) {
                if (buf[i] == delimiter) {
                    this.len = i;
                    return;
                }
            }
            this.len = len;
        }
    }

    @Override
    public void write(final byte[] b) {
        write(b, 0, b.length);
    }

    @Override
    public void write(final byte[] b, final int off, final int len) {
        int newLen = this.len + len;
        ensureCapacity(newLen);
        System.arraycopy(b, off, buf, this.len, len);
        this.len = newLen;
    }

    @Override
    public void write(final int b) {
        ensureCapacity(len + 1);
        buf[len++] = (byte) b;
    }

    public void write(final ByteBuffer bb) {
        int rem = bb.remaining();
        ensureCapacity(len + rem);
        bb.get(buf, len, rem);
        len += rem;
    }

    @Override
    public boolean isEmpty() {
        return len == 0;
    }

    @Override
    public int length() {
        return len;
    }

    public void reset() {
        len = 0;
    }

    public boolean copylessToByteArray() {
        return len == buf.length;
    }

    public byte[] toByteArray() {
        if (copylessToByteArray()) {
            return buf;
        } else {
            return Arrays.copyOf(buf, len);
        }
    }

    public ByteArrayProcessable toByteArrayProcessable() {
        return new ByteArrayProcessable(buf, 0, len);
    }

    public int count(final byte b) {
        int count = 0;
        for (int i = 0; i < len; ++i) {
            if (buf[i] == b) {
                ++count;
            }
        }
        return count;
    }

    @Override
    public <E extends Exception> void processWith(
        final VoidProcessor<? super byte[], E> processor)
        throws E
    {
        processor.process(buf, 0, len);
    }

    @Override
    public <R, E extends Exception> R processWith(
        final Processor<? super byte[], R, E> processor)
        throws E
    {
        return processor.process(buf, 0, len);
    }

    @Override
    public void process(final byte[] buf, final int off, final int len) {
        write(buf, off, len);
    }

    @Override
    public int transferTo(final byte[] buf, final int off, final int len) {
        int transferSize = Math.min(len, this.len);
        System.arraycopy(this.buf, 0, buf, off, transferSize);
        this.len -= transferSize;
        System.arraycopy(this.buf, transferSize, this.buf, 0, len);
        return transferSize;
    }
}

