package ru.yandex.io;

import java.io.IOException;
import java.io.InputStream;

public abstract class DecodingInputStream extends InputStream {
    private static final int BUFFER_SIZE = 4096;
    private static final int BYTE_MASK = 0xFF;

    protected byte[] buf = new byte[BUFFER_SIZE];
    protected final InputStream in;
    protected int buflen = 0;
    protected int pos = 0;

    protected DecodingInputStream(final InputStream in) {
        this.in = in;
    }

    protected abstract void fillBuffer() throws IOException;

    protected void ensureCapacity(final int capacity) {
        if (capacity < 0) {
            throw new IllegalArgumentException(
                "negative capacity: " + capacity);
        }
        if (capacity > buf.length) {
            int newSize = Math.max(buf.length << 1, capacity);
            buf = new byte[newSize];
        }
    }

    @Override
    public int available() throws IOException {
        return buflen - pos;
    }

    @Override
    public void close() throws IOException {
        in.close();
    }

    @Override
    public int read() throws IOException {
        if (pos >= buflen) {
            fillBuffer();
            if (pos >= buflen) {
                return -1;
            }
        }
        return buf[pos++] & BYTE_MASK;
    }

    @Override
    public int read(final byte[] b) throws IOException {
        return read(b, 0, b.length);
    }

    @Override
    public int read(final byte[] b, final int off, final int len)
        throws IOException
    {
        if (pos >= buflen) {
            fillBuffer();
            if (pos >= buflen) {
                return -1;
            }
        }
        int transfer = Math.min(buflen - pos, len);
        System.arraycopy(buf, pos, b, off, transfer);
        pos += transfer;
        return transfer;
    }
}

