package ru.yandex.io;

import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

import ru.yandex.charset.Decoder;

public final class IOStreamUtils {
    private static final int BUFFER_SIZE = 4096;

    private IOStreamUtils() {
    }

    public static void skipTo(final InputStream in, final byte[] mark)
        throws IOException
    {
        byte[] buf = new byte[mark.length];
        int len = 0;
        while (len < buf.length) {
            int read = in.read();
            if (read == -1) {
                throw new EOFException("Stream is shorter than skip mark");
            }
            buf[len++] = (byte) read;
        }

        while (!Arrays.equals(buf, mark)) {
            int read = in.read();
            if (read == -1) {
                throw new EOFException("Skip mark not found");
            }
            for (int i = 1; i < buf.length; ++i) {
                buf[i - 1] = buf[i];
            }
            buf[buf.length - 1] = (byte) read;
        }
    }

    public static void copy(final InputStream in, final OutputStream out)
        throws IOException
    {
        byte[] buf = new byte[BUFFER_SIZE];
        while (true) {
            int read = in.read(buf);
            if (read == -1) {
                break;
            }
            out.write(buf, 0, read);
        }
    }

    public static int peek(final InputStream in, final byte[] buf)
        throws IOException
    {
        int pos = 0;
        int left = buf.length;
        try (InputStreamResetter resetter =
                new InputStreamResetter(in, left))
        {
            while (left != 0) {
                int read = resetter.in().read(buf, pos, left);
                if (read == -1) {
                    break;
                }
                pos += read;
                left -= read;
            }
        }
        return pos;
    }

    public static StringBuilder consume(final Reader reader)
        throws IOException
    {
        StringBuilder sb = new StringBuilder();
        consume(reader, sb);
        return sb;
    }

    public static void consume(final Reader reader, final StringBuilder sb)
        throws IOException
    {
        char[] buf = new char[BUFFER_SIZE];
        while (true) {
            int read = reader.read(buf);
            if (read == -1) {
                break;
            }
            sb.append(buf, 0, read);
        }
    }

    public static DecodableByteArrayOutputStream consume(final InputStream in)
        throws IOException
    {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        copy(in, out);
        return out;
    }

    public static DecodableByteArrayOutputStream readResource(
        final Class<?> clazz,
        final String name)
        throws IOException
    {
        try (InputStream in = clazz.getResourceAsStream(name)) {
            if (in == null) {
                throw new FileNotFoundException(
                    "Package " + clazz.getPackageName()
                    + " resource not found: " + name);
            }
            return consume(in);
        }
    }

    public static String readResourceAsString(
        final Class<?> clazz,
        final String name)
        throws IOException
    {
        Decoder decoder = new Decoder(StandardCharsets.UTF_8);
        readResource(clazz, name).processWith(decoder);
        return decoder.toString();
    }

    public static String readResourceAsStringQuiet(
        final Class<?> clazz,
        final String name)
    {
        try {
            return readResourceAsString(clazz, name);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}

