package ru.yandex.io;

import java.io.IOException;
import java.io.InputStream;

public class QuotedPrintableInputStream extends DecodingInputStream {
    private static final int SHIFT = 4;
    private static final byte[] TABLE = {
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        0, 1, 2, 3, 4, 5, 6, 7, 8, 9, -1, -1, -1, -1, -1, -1,
        -1, 10, 11, 12, 13, 14, 15, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
        -1, 10, 11, 12, 13, 14, 15
    };

    public QuotedPrintableInputStream(final InputStream in) {
        super(in);
    }

    protected void malformed(final int read, final int c1, final int c2)
        throws IOException
    {
        throw new MalformedQuotedPrintableException(
            "Malformed char codes after equal sign: " + c1 + ',' + ' ' + c2);
    }

    protected int doRead() throws IOException {
        return in.read(buf);
    }

    @Override
    protected void fillBuffer() throws IOException {
        while (true) {
            int read = doRead();
            pos = 0;
            while (pos < read) {
                if (buf[pos] == '=') {
                    break;
                } else {
                    ++pos;
                }
            }
            buflen = pos;
            while (pos < read) {
                int c1;
                int c2;
                if (++pos < read) {
                    c1 = buf[pos++];
                    if (c1 == '\n') {
                        // Malformed soft line break
                        c2 = -1;
                    } else {
                        if (pos < read) {
                            c2 = buf[pos++];
                        } else {
                            c2 = in.read();
                        }
                    }
                } else {
                    c1 = in.read();
                    if (c1 == '\n') {
                        // Malformed soft line break
                        c2 = -1;
                    } else {
                        c2 = in.read();
                    }
                }
                if (c1 == '\r') {
                    // soft line break
                    if (c2 != '\n') {
                        malformed(read, c1, c2);
                        break;
                    }
                } else if (c1 != '\n') {
                    int upper;
                    int lower;
                    try {
                        upper = TABLE[c1];
                        lower = TABLE[c2];
                    } catch (RuntimeException e) {
                        upper = -1;
                        lower = -1;
                    }
                    if (upper >= 0 && lower >= 0) {
                        buf[buflen++] = (byte) ((upper << SHIFT) + lower);
                    } else {
                        malformed(read, c1, c2);
                        break;
                    }
                }
                while (pos < read) {
                    if (buf[pos] == '=') {
                        break;
                    } else {
                        buf[buflen++] = buf[pos++];
                    }
                }
            }
            // Check that something was decoded, input could consist only from
            // soft line breaks
            if (pos != read || buflen != 0) {
                pos = 0;
                break;
            }
        }
    }
}
