package ru.yandex.io;

import java.io.IOException;
import java.io.Writer;

public class TrimmingWriter extends Writer {
    private enum State {
        INITIAL,
        NEWLINE,
        SPACES,
        TEXT
    }

    private static final int BUFFER_SIZE = 127;

    private final char[] buf = new char[BUFFER_SIZE + 1];
    private final Writer writer;
    private int len = 0;
    private State state = State.INITIAL;

    public TrimmingWriter(final Writer writer) {
        this.writer = writer;
    }

    @Override
    public void write(final int ch) throws IOException {
        char c = (char) ch;
        if (c == '\n') {
            switch (state) {
                case INITIAL:
                case NEWLINE:
                    break;
                default: // SPACES or TEXT
                    writer.write(buf, 0, len);
                    len = 0;
                    state = State.NEWLINE;
                    break;
            }
        } else if (c <= ' ' || Character.isSpaceChar(c)) {
            switch (state) {
                case INITIAL:
                case NEWLINE:
                case SPACES:
                    break;
                default: // TEXT
                    state = State.SPACES;
                    break;
            }
        } else {
            switch (state) {
                case INITIAL:
                    buf[len++] = c;
                    state = State.TEXT;
                    break;
                case NEWLINE:
                    buf[len++] = '\n';
                    buf[len++] = c;
                    state = State.TEXT;
                    break;
                case SPACES:
                    buf[len++] = ' ';
                    buf[len++] = c;
                    state = State.TEXT;
                    break;
                default: // TEXT
                    buf[len++] = c;
                    break;
            }
            if (len >= BUFFER_SIZE) {
                writer.write(buf, 0, len);
                len = 0;
            }
        }
    }

    @Override
    public void write(final char[] cbuf, final int off, final int len)
        throws IOException
    {
        for (int i = 0; i < len; ++i) {
            write(cbuf[i + off]);
        }
    }

    @Override
    public void flush() throws IOException {
        if (len > 0) {
            writer.write(buf, 0, len);
            len = 0;
        }
        writer.flush();
    }

    @Override
    public void close() throws IOException {
        flush();
        writer.close();
    }
}

