package ru.yandex.util.ip;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;

import ru.yandex.collection.ByteArrayBitIterator;

public class Cidr implements Iterable<Integer> {
    private static final int BITS_BITS = 3;
    private static final int BYTE_BITS = 1 << BITS_BITS;
    private static final int BITS_MASK = BYTE_BITS - 1;

    private final byte[] addr;
    private final int len;

    public Cidr(final byte[] addr, final int len) {
        this.addr = addr;
        this.len = len;
    }

    public int addrLen() {
        return addr.length;
    }

    public String toBitString() {
        StringBuilder sb = new StringBuilder(len);
        ByteArrayBitIterator iter = iterator();
        for (int i = 0; i < len; ++i) {
            sb.append((char) ('0' + iter.nextInt()));
        }
        return new String(sb);
    }

    public boolean matches(final InetAddress address) {
        return matches(address.getAddress());
    }

    public boolean matches(final byte[] addr) {
        if (addr.length != this.addr.length) {
            return false;
        }
        if (len == (addr.length << BITS_BITS)) {
            return Arrays.equals(addr, this.addr);
        }
        int bytes = len >> BITS_BITS;
        for (int i = 0; i < bytes; ++i) {
            if (addr[i] != this.addr[i]) {
                return false;
            }
        }
        int rem = len & BITS_MASK;
        if (rem == 0) {
            return true;
        } else {
            int shift = BYTE_BITS - rem;
            return (addr[bytes] >> shift) == (this.addr[bytes] >> shift);
        }
    }

    @Override
    public ByteArrayBitIterator iterator() {
        return new ByteArrayBitIterator(addr, 0, len);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        try {
            sb.append(IpCompressor.compress(InetAddress.getByAddress(addr)));
        } catch (UnknownHostException e) {
            for (int i = 0; i < addr.length; ++i) {
                if (i != 0) {
                    sb.append('.');
                }
                sb.append(addr[i] & 0xff);
            }
        }
        if (len < addr.length << BITS_BITS) {
            sb.append('/');
            sb.append(len);
        }
        return new String(sb);
    }

    public static Cidr fromBitString(final int addrSize, final String str) {
        byte[] addr = new byte[addrSize];
        int pos = 0;
        int bit = 0;
        int len = str.length();
        int currentByte = 0;
        for (int i = 0; i < len; ++i) {
            currentByte <<= 1;
            if (str.charAt(i) != '0') {
                currentByte |= 1;
            }
            ++bit;
            if (bit == BYTE_BITS) {
                bit = 0;
                addr[pos++] = (byte) currentByte;
                currentByte = 0;
            }
        }
        if (bit != 0) {
            currentByte <<= BYTE_BITS - bit;
            addr[pos] = (byte) currentByte;
        }
        return new Cidr(addr, len);
    }

    public static Cidr fromString(final String ip) throws IOException {
        int slash = ip.indexOf('/');
        byte[] addr;
        int len;
        if (slash == -1) {
            addr = InetAddress.getByName(ip).getAddress();
            len = addr.length << BITS_BITS;
        } else {
            addr = InetAddress.getByName(ip.substring(0, slash)).getAddress();
            try {
                len = Integer.parseInt(ip.substring(slash + 1));
            } catch (RuntimeException e) {
                throw new IOException("Malformed CIDR <" + ip + '>', e);
            }
            if (len > (addr.length << BITS_BITS)) {
                throw new IOException("Malformed CIDR <" + ip + '>');
            }
        }
        return new Cidr(addr, len);
    }
}

