package ru.yandex.util.ip;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.nio.charset.StandardCharsets;

public interface IpSet<T> {
    public interface Builder<T> {
        void add(String ip) throws IOException;

        T build() throws IOException;
    }

    Builder<T> createBuilder();

    boolean contains(T set, InetAddress ip);

    default T load(final File file) throws IOException {
        try (InputStream in = new FileInputStream(file)) {
            return load(in);
        } catch (IOException e) {
            throw new IOException("Failed to load IPs from: " + file, e);
        }
    }

    default T load(final InputStream in) throws IOException {
        return load(in, createBuilder());
    }

    static <T> T load(final InputStream in, final Builder<T> builder)
        throws IOException
    {
        try (BufferedReader reader =
                new BufferedReader(
                    new InputStreamReader(
                        in,
                        // We could use ISO_8859_1 here for faster decoding,
                        // but using utf-8 will guard us from comments in
                        // exotic encodings like cp1251 or koi8-r
                        StandardCharsets.UTF_8)))
        {
            String line = reader.readLine();
            for (; line != null; line = reader.readLine()) {
                line = line.trim();
                if (!line.isEmpty()) {
                    boolean ignore = false;
                    switch (line.charAt(0)) {
                        case '#':
                        case '$':
                            ignore = true;
                            break;
                        case ':':
                            ignore =
                                line.length() < 2 || line.charAt(1) != ':';
                            break;
                        default:
                            break;
                    }
                    if (!ignore) {
                        try {
                            builder.add(line);
                        } catch (IOException e) {
                            throw new IOException(
                                "Failed to parse IP <" + line + '>',
                                e);
                        }
                    }
                }
            }
            return builder.build();
        }
    }
}

