#include <mail/so/libs/jniwrapper_base/jniwrapper_base.h>

#include <library/cpp/regex/hyperscan/hyperscan.h>
#include <library/cpp/regex/pcre/pcre.h>

#include <util/generic/vector.h>

#include <jni.h>

class IJniPcre {
public:
    virtual ~IJniPcre() = default;
    virtual int AddRegex(TString&& regex) = 0;
    virtual int FindMatch(TString&& str) const = 0;
};

class TJniPcre: public IJniPcre {
private:
    TVector<NPcre::TPcre<char>> Regexes;

public:
    int AddRegex(TString&& regex) override {
        int size = Regexes.size();
        Regexes.emplace_back(
            regex.c_str(),
            NPcre::EOptimize::JIT,
            PCRE_NO_AUTO_CAPTURE | PCRE_DOTALL | PCRE_UTF8);
        return size;
    }

    int FindMatch(TString&& str) const override {
        for (size_t i = 0; i < Regexes.size(); ++i) {
            if (Regexes.at(i).Matches(str)) {
                return i;
            }
        }
        return -1;
    }
};

class THyperscanCallback {
public:
    int MatchedId;

public:
    THyperscanCallback()
        : MatchedId(-1)
    {
    }

    int operator()(
        unsigned int id,
        unsigned long long /*from*/,
        unsigned long long /*to*/)
    {
        MatchedId = id;
        return HS_SCAN_TERMINATED;
    }
};

class TJniHyperscan: public IJniPcre {
private:
    TVector<TString> Regexes;
    NHyperscan::TDatabase Database;

public:
    int AddRegex(TString&& regex) override {
        int size = Regexes.size();
        Regexes.emplace_back(std::move(regex));
        TVector<const char*> strs;
        TVector<unsigned int> flags;
        TVector<unsigned int> ids;
        for (int i = 0; i <= size; ++i) {
            strs.push_back(Regexes.at(i).c_str());
            flags.push_back(
                HS_FLAG_DOTALL | HS_FLAG_UTF8 | HS_FLAG_SINGLEMATCH);
            ids.push_back(i);
        }
        Database = NHyperscan::CompileMulti(strs, flags, ids);
        return size;
    }

    int FindMatch(TString&& str) const override {
        if (Database) {
            THyperscanCallback callback;
            NHyperscan::Scan(
                Database,
                NHyperscan::MakeScratch(Database),
                str,
                callback);
            return callback.MatchedId;
        } else {
            return -1;
        }
    }
};

extern "C" JNIEXPORT jlong JNICALL
Java_ru_yandex_jni_pcre_JniPcre_createInstance(
    JNIEnv* env,
    jclass,
    jboolean useHyperscan)
{
    try {
        if (useHyperscan) {
            return reinterpret_cast<jlong>(new TJniHyperscan());
        } else {
            return reinterpret_cast<jlong>(new TJniPcre());
        }
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
        return 0;
    }
}

extern "C" JNIEXPORT void JNICALL
Java_ru_yandex_jni_pcre_JniPcre_destroyInstance(
    JNIEnv*,
    jclass,
    jlong instance)
{
    delete reinterpret_cast<IJniPcre*>(instance);
}

extern "C" JNIEXPORT jint JNICALL
Java_ru_yandex_jni_pcre_JniPcre_addRegex(
    JNIEnv* env,
    jclass,
    jlong instance,
    jstring regex)
{
    IJniPcre* pcre = reinterpret_cast<IJniPcre*>(instance);
    try {
        return pcre->AddRegex(NJniWrapper::JStringToUtf(env, regex));
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
        return -1;
    }
}

extern "C" JNIEXPORT jint JNICALL
Java_ru_yandex_jni_pcre_JniPcre_findMatch(
    JNIEnv* env,
    jclass,
    jlong instance,
    jstring str)
{
    const IJniPcre* pcre = reinterpret_cast<const IJniPcre*>(instance);
    try {
        return pcre->FindMatch(NJniWrapper::JStringToUtf(env, str));
    } catch (...) {
        NJniWrapper::RethrowAsJavaException(env);
        return -1;
    }
}

